--
-- ninja.lua
-- Utilities for generating Ninja build files
-- Author: Nick Clark
-- Copyright (c) 2025 Jess Perkins and the Premake project
--

local p = premake

p.modules.ninja = {}
p.modules.ninja._VERSION = p._VERSION

local ninja = p.modules.ninja

function ninja.esc(value)
    value = value:gsub("%$", "$$")
    value = value:gsub(":", "$:")
    value = value:gsub("\n", "$\n")
    value = value:gsub('%(', '\\(')
    value = value:gsub('%)', '\\)')
    value = value:gsub('"', '\\"')
    value = value:gsub(" ", "\\ ")
    
    return value
end

function ninja.header(target)
    local kind = iif(target.project, "project", "workspace")
    _p("# %s %s Ninja build file generated by Premake", target.name, kind)
    _p('ninja_required_version = 1.6')
    _p('')
end

function ninja.key(cfg)
    local name = cfg.project.name
    local buildcfg = cfg.buildcfg
    if cfg.platform then
        return name .. "_" .. (buildcfg or "") .. "_" .. cfg.platform
    else
        return name .. (buildcfg and ("_" .. buildcfg) or "")
    end
end

function ninja.getprjconfigfilename(prj)
    return prj.name .. ".ninja"
end

function ninja.getninjafilename(target, searchprjs)
    local count = 0
    for wks in p.global.eachWorkspace() do
        if wks.location == target.location then
            count = count + 1
        end

        if searchprjs then
            for _, prj in ipairs(wks.projects) do
                if prj.location == target.location then
                    count = count + 1
                end
            end
        end
    end

    if count == 1 then
        return "build.ninja"
    else
        return target.name .. ".ninja"
    end
end

function ninja.gettoolset(cfg)
    local default = p.action.current().toolset
    local toolset, version = p.tools.canonical(cfg.toolset or default)
    if not toolset then
        error("No toolset found for '" .. tostring(cfg.toolset) .. "'")
    end
    return toolset
end

function ninja.list(value)
    if #value > 0 then
        return " " .. table.concat(value, " ")
    else
        return ""
    end
end

-- Override tools.getrelative to use workspace-relative paths for Ninja
-- Ninja builds are always executed from the workspace root, so all paths
-- must be relative to the workspace, not the project
function ninja.getrelative(prj, value)
    if type(value) == "table" then
        local result = {}
        for i, name in ipairs(value) do
            result[i] = ninja.getrelative(prj, name)
        end
        return result
    else
        if value then
            local result = value
            if path.hasdeferredjoin(result) then
                result = path.resolvedeferredjoin(result)
            end
            -- Use workspace location instead of project location for Ninja
            return path.getrelative(prj.workspace.location, result)
        end
    end
end

include("ninja_cpp.lua")
include("ninja_workspace.lua")

return ninja
