import { Extension, Facet, SelectionRange } from '@codemirror/state';
import { ViewPlugin, ViewUpdate } from '@codemirror/view';
import type { RelativePosition, Text, Transaction, UndoManager, YTextEvent } from 'yjs';
export type ID = {
    client: number;
    clock: number;
};
export type Position = {
    type: ID | null;
    tname: string | null;
    item: ID | null;
    assoc: number;
};
export type Range = {
    yanchor: Position;
    yhead: Position;
};
/**
 * Defines a range on text using relative positions that can be transformed back to
 * absolute positions. (https://docs.yjs.dev/api/relative-positions)
 */
export declare class YRange {
    yanchor: RelativePosition;
    yhead: RelativePosition;
    /**
     * @param yanchor
     * @param yhead
     */
    constructor(yanchor: RelativePosition, yhead: RelativePosition);
    /**
     * Convert the position to JSON
     */
    toJSON(): Range;
    /**
     * Convert a JSON range to a YRange
     * @param json Range to convert
     * @return The range as YRange
     */
    static fromJSON(json: Range): YRange;
}
/**
 * Yjs binding configuration
 */
export declare class YSyncConfig {
    ytext: Text;
    /**
     * Create a new binding configuration
     *
     * @param ytext Yjs text to synchronize
     */
    constructor(ytext: Text);
    /**
     * Helper function to transform an absolute index position to a Yjs-based relative position
     * (https://docs.yjs.dev/api/relative-positions).
     *
     * A relative position can be transformed back to an absolute position even after the document has changed. The position is
     * automatically adapted. This does not require any position transformations. Relative positions are computed based on
     * the internal Yjs document model. Peers that share content through Yjs are guaranteed that their positions will always
     * synced up when using relative positions.
     *
     * ```js
     * import { ySyncFacet } from 'y-codemirror'
     *
     * ..
     * const ysync = view.state.facet(ySyncFacet)
     * // transform an absolute index position to a ypos
     * const ypos = ysync.getYPos(3)
     * // transform the ypos back to an absolute position
     * ysync.fromYPos(ypos) // => 3
     * ```
     *
     * It cannot be guaranteed that absolute index positions can be synced up between peers.
     * This might lead to undesired behavior when implementing features that require that all peers see the
     * same marked range (e.g. a comment plugin).
     *
     * @param pos
     * @param assoc
     */
    toYPos(pos: number, assoc?: number): RelativePosition;
    /**
     * @param rpos
     */
    fromYPos(rpos: RelativePosition | Record<string, any>): {
        pos: number;
        assoc: number;
    };
    /**
     * @param range
     * @return
     */
    toYRange(range: SelectionRange): YRange;
    /**
     * @param yrange
     */
    fromYRange(yrange: YRange): SelectionRange;
}
/**
 * Yjs binding facet
 */
export declare const ySyncFacet: Facet<YSyncConfig, YSyncConfig>;
/**
 * Yjs binding annotation
 *
 * It is used to track the origin of the document changes
 */
export declare const ySyncAnnotation: import("@codemirror/state").AnnotationType<YSyncConfig>;
/**
 * Yjs binding view plugin to synchronize the
 * editor state with the Yjs document.
 */
export declare const ySync: ViewPlugin<{
    update(update: ViewUpdate): void;
    destroy(): void;
    conf: YSyncConfig;
    _observer: (event: YTextEvent, tr: Transaction) => void;
    _ytext: Text;
}>;
/**
 * Extension for CodeMirror 6 binding the Yjs text (source of truth)
 * and the editor state.
 *
 * @param ytext Yjs text to bind
 * @param undoManager Yjs text undo manager
 * @returns CodeMirror 6 extension
 */
export declare function ybinding({ ytext, undoManager }: {
    ytext: Text;
    undoManager?: UndoManager;
}): Extension;
