import collections

""" Mapping of CUDA functions, include files, constants, and types to ROCm/HIP equivalents """

# List of math functions that should be replaced inside device code only.
MATH_TRANSPILATIONS = collections.OrderedDict([
    ("std::max", ("::max")),
    ("std::min", ("::min")),
    ("std::ceil", ("::ceil")),
    ("std::floor", ("::floor")),
    ("std::exp", ("::exp")),
    ("std::log", ("::log")),
    ("std::pow", ("::pow")),
    ("std::fabs", ("::fabs")),
    ("std::fmod", ("::fmod")),
    ("std::remainder", ("::remainder")),
    ("std::frexp", ("::frexp")),
])

CUDA_TYPE_NAME_MAP = collections.OrderedDict([
    ("CUresult", "hipError_t"),
    ("cudaError_t", "hipError_t"),
    ("cudaError", "hipError_t"),
    ("CUDA_ARRAY3D_DESCRIPTOR", "HIP_ARRAY3D_DESCRIPTOR"),
    ("CUDA_ARRAY_DESCRIPTOR", "HIP_ARRAY_DESCRIPTOR"),
    ("CUDA_MEMCPY2D", "hip_Memcpy2D"),
    ("CUDA_MEMCPY3D", "HIP_MEMCPY3D"),
    ("CUDA_MEMCPY3D_PEER", "HIP_MEMCPY3D_PEER"),
    ("CUDA_POINTER_ATTRIBUTE_P2P_TOKENS", "HIP_POINTER_ATTRIBUTE_P2P_TOKENS"),
    ("CUDA_RESOURCE_DESC", "HIP_RESOURCE_DESC"),
    ("CUDA_RESOURCE_VIEW_DESC", "HIP_RESOURCE_VIEW_DESC"),
    ("CUipcEventHandle", "hipIpcEventHandle"),
    ("CUipcMemHandle", "hipIpcMemHandle"),
    ("CUaddress_mode", "hipAddress_mode"),
    ("CUarray_cubemap_face", "hipArray_cubemap_face"),
    ("CUarray_format", "hipArray_format"),
    ("CUcomputemode", "hipComputemode"),
    ("CUmem_advise", "hipMemAdvise"),
    ("CUmem_range_attribute", "hipMemRangeAttribute"),
    ("CUctx_flags", "hipCctx_flags"),
    ("CUdevice", "hipDevice_t"),
    ("CUdevice_attribute_enum", "hipDeviceAttribute_t"),
    ("CUdevice_attribute", "hipDeviceAttribute_t"),
    ("CUpointer_attribute", "hipPointer_attribute"),
    ("CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL", "HIP_POINTER_ATTRIBUTE_DEVICE_ORDINAL"),
    ("CU_POINTER_ATTRIBUTE_BUFFER_ID", "HIP_POINTER_ATTRIBUTE_BUFFER_ID"),
    ("CUdeviceptr", "hipDeviceptr_t"),
    ("CUarray_st", "hipArray"),
    ("CUarray", "hipArray *"),
    ("CUdevprop_st", "hipDeviceProp_t"),
    ("CUdevprop", "hipDeviceProp_t"),
    ("CUfunction", "hipFunction_t"),
    ("CUgraphicsResource", "hipGraphicsResource_t"),
    ("CUmipmappedArray", "hipMipmappedArray_t"),
    ("CUfunction_attribute", "hipFuncAttribute_t"),
    ("CUfunction_attribute_enum", "hipFuncAttribute_t"),
    ("CUgraphicsMapResourceFlags", "hipGraphicsMapFlags"),
    ("CUgraphicsMapResourceFlags_enum", "hipGraphicsMapFlags"),
    ("CUgraphicsRegisterFlags", "hipGraphicsRegisterFlags"),
    ("CUgraphicsRegisterFlags_enum", "hipGraphicsRegisterFlags"),
    ("CUoccupancy_flags", "hipOccupancyFlags"),
    ("CUoccupancy_flags_enum", "hipOccupancyFlags"),
    ("CUfunc_cache_enum", "hipFuncCache"),
    ("CUfunc_cache", "hipFuncCache"),
    ("CUipcMem_flags", "hipIpcMemFlags"),
    ("CUipcMem_flags_enum", "hipIpcMemFlags"),
    ("CUjit_cacheMode", "hipJitCacheMode"),
    ("CUjit_cacheMode_enum", "hipJitCacheMode"),
    ("CUjit_fallback", "hipJitFallback"),
    ("CUjit_fallback_enum", "hipJitFallback"),
    ("CUjit_option", "hipJitOption"),
    ("CUjit_option_enum", "hipJitOption"),
    ("CUjit_target", "hipJitTarget"),
    ("CUjit_target_enum", "hipJitTarget"),
    ("CUjitInputType", "hipJitInputType"),
    ("CUjitInputType_enum", "hipJitInputType"),
    ("CUlimit", "hipLimit_t"),
    ("CUlimit_enum", "hipLimit_t"),
    ("CUmemAttach_flags", "hipMemAttachFlags_t"),
    ("CUmemAttach_flags_enum", "hipMemAttachFlags_t"),
    ("CUmemorytype", "hipMemType_t"),
    ("CUmemorytype_enum", "hipMemType_t"),
    ("CUresourcetype", "hipResourceType"),
    ("CUresourcetype_enum", "hipResourceType"),
    ("CUresourceViewFormat", "hipResourceViewFormat"),
    ("CUresourceViewFormat_enum", "hipResourceViewFormat"),
    ("CUsharedconfig", "hipSharedMemConfig"),
    ("CUsharedconfig_enum", "hipSharedMemConfig"),
    ("CUcontext", "hipCtx_t"),
    ("CUmodule", "hipModule_t"),
    ("CUstream", "hipStream_t"),
    ("CUstream_st", "ihipStream_t"),
    ("CUstreamCallback", "hipStreamCallback_t"),
    ("CUsurfObject", "hipSurfaceObject"),
    ("CUsurfref", "hipSurfaceReference_t"),
    ("CUtexObject", "hipTextureObject_t"),
    ("CUtexref", "textureReference"),
    ("CUstream_flags", "hipStreamFlags"),
    ("CUstreamWaitValue_flags", "hipStreamWaitValueFlags"),
    ("CUstreamWriteValue_flags", "hipStreamWriteValueFlags"),
    ("CUstreamBatchMemOpType", "hipStreamBatchMemOpType"),
    ("CUdevice_P2PAttribute", "hipDeviceP2PAttribute"),
    ("CUevent", "hipEvent_t"),
    ("CUevent_st", "ihipEvent_t"),
    ("CUevent_flags", "hipEventFlags"),
    ("CUfilter_mode", "hipTextureFilterMode"),
    ("CUGLDeviceList", "hipGLDeviceList"),
    ("CUGLmap_flags", "hipGLMapFlags"),
    ("CUd3d9DeviceList", "hipD3D9DeviceList"),
    ("CUd3d9map_flags", "hipD3D9MapFlags"),
    ("CUd3d9register_flags", "hipD3D9RegisterFlags"),
    ("CUd3d10DeviceList", "hipd3d10DeviceList"),
    ("CUd3d10map_flags", "hipD3D10MapFlags"),
    ("CUd3d10register_flags", "hipD3D10RegisterFlags"),
    ("CUd3d11DeviceList", "hipd3d11DeviceList"),
    ("CUeglStreamConnection_st", "hipEglStreamConnection"),
    ("CUeglStreamConnection", "hipEglStreamConnection"),
    ("libraryPropertyType_t", "hipLibraryPropertyType_t"),
    ("libraryPropertyType", "hipLibraryPropertyType_t"),
    ("cudaStreamCallback_t", "hipStreamCallback_t"),
    ("cudaArray", "hipArray"),
    ("cudaArray_t", "hipArray_t"),
    ("cudaArray_const_t", "hipArray_const_t"),
    ("cudaMipmappedArray_t", "hipMipmappedArray_t"),
    ("cudaMipmappedArray_const_t", "hipMipmappedArray_const_t"),
    ("cudaArrayDefault", "hipArrayDefault"),
    ("cudaArrayLayered", "hipArrayLayered"),
    ("cudaArraySurfaceLoadStore", "hipArraySurfaceLoadStore"),
    ("cudaArrayCubemap", "hipArrayCubemap"),
    ("cudaArrayTextureGather", "hipArrayTextureGather"),
    ("cudaMemoryAdvise", "hipMemoryAdvise"),
    ("cudaMemRangeAttribute", "hipMemRangeAttribute"),
    ("cudaMemcpyKind", "hipMemcpyKind"),
    ("cudaMemoryType", "hipMemoryType"),
    ("cudaExtent", "hipExtent"),
    ("cudaPitchedPtr", "hipPitchedPtr"),
    ("cudaPos", "hipPos"),
    ("cudaEvent_t", "hipEvent_t"),
    ("cudaStream_t", "hipStream_t"),
    ("cudaPointerAttributes", "hipPointerAttribute_t"),
    ("cudaDeviceAttr", "hipDeviceAttribute_t"),
    ("cudaDeviceProp", "hipDeviceProp_t"),
    ("cudaDeviceP2PAttr", "hipDeviceP2PAttribute"),
    ("cudaComputeMode", "hipComputeMode"),
    ("cudaFuncCache", "hipFuncCache_t"),
    ("cudaFuncAttributes", "hipFuncAttributes"),
    ("cudaSharedMemConfig", "hipSharedMemConfig"),
    ("cudaLimit", "hipLimit_t"),
    ("cudaOutputMode", "hipOutputMode"),
    ("cudaTextureReadMode", "hipTextureReadMode"),
    ("cudaTextureFilterMode", "hipTextureFilterMode"),
    ("cudaChannelFormatKind", "hipChannelFormatKind"),
    ("cudaChannelFormatDesc", "hipChannelFormatDesc"),
    ("cudaResourceDesc", "hipResourceDesc"),
    ("cudaResourceViewDesc", "hipResourceViewDesc"),
    ("cudaTextureDesc", "hipTextureDesc"),
    ("surfaceReference", "hipSurfaceReference"),
    ("cudaTextureObject_t", "hipTextureObject_t"),
    ("cudaResourceType", "hipResourceType"),
    ("cudaResourceViewFormat", "hipResourceViewFormat"),
    ("cudaTextureAddressMode", "hipTextureAddressMode"),
    ("cudaSurfaceBoundaryMode", "hipSurfaceBoundaryMode"),
    ("cudaSurfaceFormatMode", "hipSurfaceFormatMode"),
    ("cudaSurfaceObject_t", "hipSurfaceObject_t"),
    ("cudaTextureType1D", "hipTextureType1D"),
    ("cudaTextureType2D", "hipTextureType2D"),
    ("cudaTextureType3D", "hipTextureType3D"),
    ("cudaTextureTypeCubemap", "hipTextureTypeCubemap"),
    ("cudaTextureType1DLayered", "hipTextureType1DLayered"),
    ("cudaTextureType2DLayered", "hipTextureType2DLayered"),
    ("cudaTextureTypeCubemapLayered", "hipTextureTypeCubemapLayered"),
    ("cudaUUID_t", "hipUUID"),
    ("cudaIpcEventHandle_t", "hipIpcEventHandle_t"),
    ("cudaIpcEventHandle_st", "hipIpcEventHandle_t"),
    ("cudaIpcMemHandle_t", "hipIpcMemHandle_t"),
    ("cudaIpcMemHandle_st", "hipIpcMemHandle_t"),
    ("cudaGraphicsCubeFace", "hipGraphicsCubeFace"),
    ("cudaGraphicsMapFlags", "hipGraphicsMapFlags"),
    ("cudaGraphicsRegisterFlags", "hipGraphicsRegisterFlags"),
    ("cudaGLDeviceList", "hipGLDeviceList"),
    ("cudaGLMapFlags", "hipGLMapFlags"),
    ("cudaD3D9DeviceList", "hipD3D9DeviceList"),
    ("cudaD3D9MapFlags", "hipD3D9MapFlags"),
    ("cudaD3D9RegisterFlags", "hipD3D9RegisterFlags"),
    ("cudaD3D10DeviceList", "hipd3d10DeviceList"),
    ("cudaD3D10MapFlags", "hipD3D10MapFlags"),
    ("cudaD3D10RegisterFlags", "hipD3D10RegisterFlags"),
    ("cudaD3D11DeviceList", "hipd3d11DeviceList"),
    ("cudaEglStreamConnection", "hipEglStreamConnection"),
    ("cublasHandle_t", "hipblasHandle_t"),
    ("cublasOperation_t", "hipblasOperation_t"),
    ("cublasStatus_t", "hipblasStatus_t"),
    ("cublasFillMode_t", "hipblasFillMode_t"),
    ("cublasDiagType_t", "hipblasDiagType_t"),
    ("cublasSideMode_t", "hipblasSideMode_t"),
    ("cublasPointerMode_t", "hipblasPointerMode_t"),
    ("cublasGemmAlgo_t", "hipblasGemmAlgo_t"),
    ("cublasAtomicsMode_t", "hipblasAtomicsMode_t"),
    ("cublasDataType_t", "hipblasDatatype_t"),
    ("curandStatus", "hiprandStatus_t"),
    ("curandStatus_t", "hiprandStatus_t"),
    ("curandRngType", "hiprandRngType_t"),
    ("curandRngType_t", "hiprandRngType_t"),
    ("curandGenerator_st", "hiprandGenerator_st"),
    ("curandGenerator_t", "hiprandGenerator_t"),
    ("curandDirectionVectorSet", "hiprandDirectionVectorSet_t"),
    ("curandDirectionVectorSet_t", "hiprandDirectionVectorSet_t"),
    ("curandOrdering", "hiprandOrdering_t"),
    ("curandOrdering_t", "hiprandOrdering_t"),
    ("curandDistribution_st", "hiprandDistribution_st"),
    ("curandHistogramM2V_st", "hiprandDistribution_st"),
    ("curandDistribution_t", "hiprandDistribution_t"),
    ("curandHistogramM2V_t", "hiprandDistribution_t"),
    ("curandDistributionShift_st", "hiprandDistributionShift_st"),
    ("curandDistributionShift_t", "hiprandDistributionShift_t"),
    ("curandDistributionM2Shift_st", "hiprandDistributionM2Shift_st"),
    ("curandDistributionM2Shift_t", "hiprandDistributionM2Shift_t"),
    ("curandHistogramM2_st", "hiprandHistogramM2_st"),
    ("curandHistogramM2_t", "hiprandHistogramM2_t"),
    ("curandHistogramM2K_st", "hiprandHistogramM2K_st"),
    ("curandHistogramM2K_t", "hiprandHistogramM2K_t"),
    ("curandDiscreteDistribution_st", "hiprandDiscreteDistribution_st"),
    ("curandDiscreteDistribution_t", "hiprandDiscreteDistribution_t"),
    ("curandMethod", "hiprandMethod_t"),
    ("curandMethod_t", "hiprandMethod_t"),
    ("curandDirectionVectors32_t", "hiprandDirectionVectors32_t"),
    ("curandDirectionVectors64_t", "hiprandDirectionVectors64_t"),
    ("curandStateMtgp32_t", "hiprandStateMtgp32_t"),
    ("curandStateMtgp32", "hiprandStateMtgp32_t"),
    ("curandStateScrambledSobol64_t", "hiprandStateScrambledSobol64_t"),
    ("curandStateSobol64_t", "hiprandStateSobol64_t"),
    ("curandStateScrambledSobol32_t", "hiprandStateScrambledSobol32_t"),
    ("curandStateSobol32_t", "hiprandStateSobol32_t"),
    ("curandStateMRG32k3a_t", "hiprandStateMRG32k3a_t"),
    ("curandStatePhilox4_32_10_t", "hiprandStatePhilox4_32_10_t"),
    ("curandStateXORWOW_t", "hiprandStateXORWOW_t"),
    ("curandState_t", "hiprandState_t"),
    ("curandState", "hiprandState_t"),
    ("CUuuid", "hipUUID"),
    ("cudaGraph_t", "hipGraph_t"),
    ("cudaGraphExec_t", "hipGraphExec_t"),
    ("__nv_bfloat16", "__hip_bfloat16"),
    ("__nv_bfloat162", "__hip_bfloat162"),
])

CUDA_INCLUDE_MAP = collections.OrderedDict([
    ("include <cuda.h", "include <hip/hip_runtime.h"),
    ('include "cuda.h', 'include "hip/hip_runtime.h'),
    ("cuda_runtime.h", "hip/hip_runtime.h"),
    ("cuda_runtime_api.h", "hip/hip_runtime_api.h"),
    ("cuda_profiler_api.h", "hip/hip_runtime_api.h"),
    ("channel_descriptor.h", "hip/channel_descriptor.h"),
    ("device_functions.h", "hip/device_functions.h"),
    ("driver_types.h", "hip/driver_types.h"),
    ("library_types.h", "hip/library_types.h"),
    ("cuComplex.h", "hip/hip_complex.h"),
    ("cuda_fp16.h", "hip/hip_fp16.h"),
    ("cuda_bf16.h", "hip/hip_bf16.h"),
    ("cuda_texture_types.h", "hip/hip_texture_types.h"),
    ("cooperative_groups.h", "hip/hip_cooperative_groups.h"),
    ("vector_types.h", "hip/hip_vector_types.h"),
    ("cublas.h", "hipblas/hipblas.h"),
    ("cublas_v2.h", "hipblas/hipblas.h"),
    ("cublasLt.h", "hipblaslt/hipblaslt.h"),
    ("curand.h", "hiprand/hiprand.h"),
    ("curand_kernel.h", "hiprand/hiprand_kernel.h"),
    ("curand_discrete.h", "hiprand/hiprand_kernel.h"),
    ("curand_discrete2.h", "hiprand/hiprand_kernel.h"),
    ("curand_globals.h", "hiprand/hiprand_kernel.h"),
    ("curand_lognormal.h", "hiprand/hiprand_kernel.h"),
    ("curand_mrg32k3a.h", "hiprand/hiprand_kernel.h"),
    ("curand_mtgp32.h", "hiprand/hiprand_kernel.h"),
    ("curand_mtgp32_host.h", "hiprand/hiprand_mtgp32_host.h"),
    ("curand_mtgp32_kernel.h", "hiprand/hiprand_kernel.h"),
    ("curand_mtgp32dc_p_11213.h", "rocrand/rocrand_mtgp32_11213.h"),
    ("curand_normal.h", "hiprand/hiprand_kernel.h"),
    ("curand_normal_static.h", "hiprand/hiprand_kernel.h"),
    ("curand_philox4x32_x.h", "hiprand/hiprand_kernel.h"),
    ("curand_poisson.h", "hiprand/hiprand_kernel.h"),
    ("curand_precalc.h", "hiprand/hiprand_kernel.h"),
    ("curand_uniform.h", "hiprand/hiprand_kernel.h"),
    ("cusparse.h", "hipsparse/hipsparse.h"),
    ("cufft.h", "hipfft/hipfft.h"),
    ("cufftXt.h", "hipfft/hipfftXt.h"),
    ("<nccl.h>", "<rccl/rccl.h>"),
    ("nvrtc.h", "hip/hiprtc.h"),
    ("thrust/system/cuda", "thrust/system/hip"),
    ("cub/util_allocator.cuh", "hipcub/hipcub.hpp"),
    ("cub/block/block_reduce.cuh", "hipcub/hipcub.hpp"),
    ("cub/block/block_raking_layout.cuh", "hipcub/hipcub.hpp"),
    ("cub/cub.cuh", "hipcub/hipcub.hpp"),
    ("cub/config.cuh", "hipcub/hipcub.hpp"),
    ("cub/util_ptx.cuh", "hipcub/hipcub.hpp"),
    ("cub/util_type.cuh", "hipcub/hipcub.hpp"),
    ("cub/device/device_run_length_encode.cuh", "hipcub/hipcub.hpp"),
    ("cub/block/block_load.cuh", "hipcub/hipcub.hpp"),
    ("cub/block/block_store.cuh", "hipcub/hipcub.hpp"),
    ("cub/block/block_scan.cuh", "hipcub/hipcub.hpp"),
    ("cub/device/device_radix_sort.cuh", "hipcub/hipcub.hpp"),
    ("cub/device/device_reduce.cuh", "hipcub/hipcub.hpp"),
    ("cub/device/device_scan.cuh", "hipcub/hipcub.hpp"),
    ("cub/device/device_select.cuh", "hipcub/hipcub.hpp"),
    ("nvtx3/nvtx3.hpp", "roctracer/roctx.h"),
    ("nvToolsExt.h", "roctracer/roctx.h"),
    ("nvml.h", "rocm_smi/rocm_smi.h"),
    ("tensorpipe/tensorpipe_cuda.h", "tensorpipe/tensorpipe_hip.h"),
])

CUDA_IDENTIFIER_MAP = collections.OrderedDict([
    ("__CUDACC__", "__HIPCC__"),
    ("CUDA_ERROR_INVALID_CONTEXT", "hipErrorInvalidContext"),
    ("CUDA_ERROR_CONTEXT_ALREADY_CURRENT", "hipErrorContextAlreadyCurrent"),
    ("CUDA_ERROR_ARRAY_IS_MAPPED", "hipErrorArrayIsMapped"),
    ("CUDA_ERROR_ALREADY_MAPPED", "hipErrorAlreadyMapped"),
    ("CUDA_ERROR_ALREADY_ACQUIRED", "hipErrorAlreadyAcquired"),
    ("CUDA_ERROR_NOT_MAPPED", "hipErrorNotMapped"),
    ("CUDA_ERROR_NOT_MAPPED_AS_ARRAY", "hipErrorNotMappedAsArray"),
    ("CUDA_ERROR_NOT_MAPPED_AS_POINTER", "hipErrorNotMappedAsPointer"),
    ("CUDA_ERROR_CONTEXT_ALREADY_IN_USE", "hipErrorContextAlreadyInUse"),
    ("CUDA_ERROR_INVALID_SOURCE", "hipErrorInvalidSource"),
    ("CUDA_ERROR_FILE_NOT_FOUND", "hipErrorFileNotFound"),
    ("CUDA_ERROR_NOT_FOUND", "hipErrorNotFound"),
    ("CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING", "hipErrorLaunchIncompatibleTexturing"),
    ("CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE", "hipErrorPrimaryContextActive"),
    ("CUDA_ERROR_CONTEXT_IS_DESTROYED", "hipErrorContextIsDestroyed"),
    ("CUDA_ERROR_NOT_PERMITTED", "hipErrorNotPermitted"),
    ("CUDA_ERROR_NOT_SUPPORTED", "hipErrorNotSupported"),
    ("cudaErrorMissingConfiguration", "hipErrorMissingConfiguration"),
    ("cudaErrorPriorLaunchFailure", "hipErrorPriorLaunchFailure"),
    ("cudaErrorInvalidDeviceFunction", "hipErrorInvalidDeviceFunction"),
    ("cudaErrorInvalidConfiguration", "hipErrorInvalidConfiguration"),
    ("cudaErrorInvalidPitchValue", "hipErrorInvalidPitchValue"),
    ("cudaErrorInvalidSymbol", "hipErrorInvalidSymbol"),
    ("cudaErrorInvalidHostPointer", "hipErrorInvalidHostPointer"),
    ("cudaErrorInvalidDevicePointer", "hipErrorInvalidDevicePointer"),
    ("cudaErrorInvalidTexture", "hipErrorInvalidTexture"),
    ("cudaErrorInvalidTextureBinding", "hipErrorInvalidTextureBinding"),
    ("cudaErrorInvalidChannelDescriptor", "hipErrorInvalidChannelDescriptor"),
    ("cudaErrorInvalidMemcpyDirection", "hipErrorInvalidMemcpyDirection"),
    ("cudaErrorAddressOfConstant", "hipErrorAddressOfConstant"),
    ("cudaErrorTextureFetchFailed", "hipErrorTextureFetchFailed"),
    ("cudaErrorTextureNotBound", "hipErrorTextureNotBound"),
    ("cudaErrorSynchronizationError", "hipErrorSynchronizationError"),
    ("cudaErrorInvalidFilterSetting", "hipErrorInvalidFilterSetting"),
    ("cudaErrorInvalidNormSetting", "hipErrorInvalidNormSetting"),
    ("cudaErrorMixedDeviceExecution", "hipErrorMixedDeviceExecution"),
    ("cudaErrorNotYetImplemented", "hipErrorNotYetImplemented"),
    ("cudaErrorMemoryValueTooLarge", "hipErrorMemoryValueTooLarge"),
    ("cudaErrorInsufficientDriver", "hipErrorInsufficientDriver"),
    ("cudaErrorSetOnActiveProcess", "hipErrorSetOnActiveProcess"),
    ("cudaErrorInvalidSurface", "hipErrorInvalidSurface"),
    ("cudaErrorDuplicateVariableName", "hipErrorDuplicateVariableName"),
    ("cudaErrorDuplicateTextureName", "hipErrorDuplicateTextureName"),
    ("cudaErrorDuplicateSurfaceName", "hipErrorDuplicateSurfaceName"),
    ("cudaErrorDevicesUnavailable", "hipErrorDevicesUnavailable"),
    ("cudaErrorIncompatibleDriverContext", "hipErrorIncompatibleDriverContext"),
    ("cudaErrorDeviceAlreadyInUse", "hipErrorDeviceAlreadyInUse"),
    ("cudaErrorLaunchMaxDepthExceeded", "hipErrorLaunchMaxDepthExceeded"),
    ("cudaErrorLaunchFileScopedTex", "hipErrorLaunchFileScopedTex"),
    ("cudaErrorLaunchFileScopedSurf", "hipErrorLaunchFileScopedSurf"),
    ("cudaErrorSyncDepthExceeded", "hipErrorSyncDepthExceeded"),
    ("cudaErrorLaunchPendingCountExceeded", "hipErrorLaunchPendingCountExceeded"),
    ("cudaErrorNotPermitted", "hipErrorNotPermitted"),
    ("cudaErrorNotSupported", "hipErrorNotSupported"),
    ("cudaErrorStartupFailure", "hipErrorStartupFailure"),
    ("cudaErrorApiFailureBase", "hipErrorApiFailureBase"),
    ("CUDA_SUCCESS", "hipSuccess"),
    ("cudaSuccess", "hipSuccess"),
    ("CUDA_ERROR_INVALID_VALUE", "hipErrorInvalidValue"),
    ("cudaErrorInvalidValue", "hipErrorInvalidValue"),
    ("CUDA_ERROR_OUT_OF_MEMORY", "hipErrorMemoryAllocation"),
    ("cudaErrorMemoryAllocation", "hipErrorMemoryAllocation"),
    ("CUDA_ERROR_NOT_INITIALIZED", "hipErrorNotInitialized"),
    ("cudaErrorInitializationError", "hipErrorInitializationError"),
    ("CUDA_ERROR_DEINITIALIZED", "hipErrorDeinitialized"),
    ("cudaErrorCudartUnloading", "hipErrorDeinitialized"),
    ("CUDA_ERROR_PROFILER_DISABLED", "hipErrorProfilerDisabled"),
    ("cudaErrorProfilerDisabled", "hipErrorProfilerDisabled"),
    ("CUDA_ERROR_PROFILER_NOT_INITIALIZED", "hipErrorProfilerNotInitialized"),
    ("cudaErrorProfilerNotInitialized", "hipErrorProfilerNotInitialized"),
    ("CUDA_ERROR_PROFILER_ALREADY_STARTED", "hipErrorProfilerAlreadyStarted"),
    ("cudaErrorProfilerAlreadyStarted", "hipErrorProfilerAlreadyStarted"),
    ("CUDA_ERROR_PROFILER_ALREADY_STOPPED", "hipErrorProfilerAlreadyStopped"),
    ("cudaErrorProfilerAlreadyStopped", "hipErrorProfilerAlreadyStopped"),
    ("CUDA_ERROR_NO_DEVICE", "hipErrorNoDevice"),
    ("cudaErrorNoDevice", "hipErrorNoDevice"),
    ("CUDA_ERROR_INVALID_DEVICE", "hipErrorInvalidDevice"),
    ("cudaErrorInvalidDevice", "hipErrorInvalidDevice"),
    ("CUDA_ERROR_INVALID_IMAGE", "hipErrorInvalidImage"),
    ("cudaErrorInvalidKernelImage", "hipErrorInvalidImage"),
    ("CUDA_ERROR_MAP_FAILED", "hipErrorMapFailed"),
    ("cudaErrorMapBufferObjectFailed", "hipErrorMapFailed"),
    ("CUDA_ERROR_UNMAP_FAILED", "hipErrorUnmapFailed"),
    ("cudaErrorUnmapBufferObjectFailed", "hipErrorUnmapFailed"),
    ("CUDA_ERROR_NO_BINARY_FOR_GPU", "hipErrorNoBinaryForGpu"),
    ("cudaErrorNoKernelImageForDevice", "hipErrorNoBinaryForGpu"),
    ("CUDA_ERROR_ECC_UNCORRECTABLE", "hipErrorECCNotCorrectable"),
    ("cudaErrorECCUncorrectable", "hipErrorECCNotCorrectable"),
    ("CUDA_ERROR_UNSUPPORTED_LIMIT", "hipErrorUnsupportedLimit"),
    ("cudaErrorUnsupportedLimit", "hipErrorUnsupportedLimit"),
    ("CUDA_ERROR_PEER_ACCESS_UNSUPPORTED", "hipErrorPeerAccessUnsupported"),
    ("cudaErrorPeerAccessUnsupported", "hipErrorPeerAccessUnsupported"),
    ("CUDA_ERROR_INVALID_PTX", "hipErrorInvalidKernelFile"),
    ("cudaErrorInvalidPtx", "hipErrorInvalidKernelFile"),
    ("CUDA_ERROR_INVALID_GRAPHICS_CONTEXT", "hipErrorInvalidGraphicsContext"),
    ("cudaErrorInvalidGraphicsContext", "hipErrorInvalidGraphicsContext"),
    ("CUDA_ERROR_NVLINK_UNCORRECTABLE", "hipErrorNvlinkUncorrectable"),
    ("cudaErrorNvlinkUncorrectable", "hipErrorNvlinkUncorrectable"),
    ("CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND", "hipErrorSharedObjectSymbolNotFound"),
    ("cudaErrorSharedObjectSymbolNotFound", "hipErrorSharedObjectSymbolNotFound"),
    ("CUDA_ERROR_SHARED_OBJECT_INIT_FAILED", "hipErrorSharedObjectInitFailed"),
    ("cudaErrorSharedObjectInitFailed", "hipErrorSharedObjectInitFailed"),
    ("CUDA_ERROR_OPERATING_SYSTEM", "hipErrorOperatingSystem"),
    ("cudaErrorOperatingSystem", "hipErrorOperatingSystem"),
    ("CUDA_ERROR_INVALID_HANDLE", "hipErrorInvalidResourceHandle"),
    ("cudaErrorInvalidResourceHandle", "hipErrorInvalidResourceHandle"),
    ("CUDA_ERROR_NOT_READY", "hipErrorNotReady"),
    ("cudaErrorNotReady", "hipErrorNotReady"),
    ("CUDA_ERROR_ILLEGAL_ADDRESS", "hipErrorIllegalAddress"),
    ("cudaErrorIllegalAddress", "hipErrorIllegalAddress"),
    ("CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES", "hipErrorLaunchOutOfResources"),
    ("cudaErrorLaunchOutOfResources", "hipErrorLaunchOutOfResources"),
    ("CUDA_ERROR_LAUNCH_TIMEOUT", "hipErrorLaunchTimeOut"),
    ("cudaErrorLaunchTimeout", "hipErrorLaunchTimeOut"),
    ("CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED", "hipErrorPeerAccessAlreadyEnabled"),
    ("cudaErrorPeerAccessAlreadyEnabled", "hipErrorPeerAccessAlreadyEnabled"),
    ("CUDA_ERROR_PEER_ACCESS_NOT_ENABLED", "hipErrorPeerAccessNotEnabled"),
    ("cudaErrorPeerAccessNotEnabled", "hipErrorPeerAccessNotEnabled"),
    ("CUDA_ERROR_ASSERT", "hipErrorAssert"),
    ("cudaErrorAssert", "hipErrorAssert"),
    ("CUDA_ERROR_TOO_MANY_PEERS", "hipErrorTooManyPeers"),
    ("cudaErrorTooManyPeers", "hipErrorTooManyPeers"),
    ("CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED", "hipErrorHostMemoryAlreadyRegistered"),
    ("cudaErrorHostMemoryAlreadyRegistered", "hipErrorHostMemoryAlreadyRegistered"),
    ("CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED", "hipErrorHostMemoryNotRegistered"),
    ("cudaErrorHostMemoryNotRegistered", "hipErrorHostMemoryNotRegistered"),
    ("CUDA_ERROR_HARDWARE_STACK_ERROR", "hipErrorHardwareStackError"),
    ("cudaErrorHardwareStackError", "hipErrorHardwareStackError"),
    ("CUDA_ERROR_ILLEGAL_INSTRUCTION", "hipErrorIllegalInstruction"),
    ("cudaErrorIllegalInstruction", "hipErrorIllegalInstruction"),
    ("CUDA_ERROR_MISALIGNED_ADDRESS", "hipErrorMisalignedAddress"),
    ("cudaErrorMisalignedAddress", "hipErrorMisalignedAddress"),
    ("CUDA_ERROR_INVALID_ADDRESS_SPACE", "hipErrorInvalidAddressSpace"),
    ("cudaErrorInvalidAddressSpace", "hipErrorInvalidAddressSpace"),
    ("CUDA_ERROR_INVALID_PC", "hipErrorInvalidPc"),
    ("cudaErrorInvalidPc", "hipErrorInvalidPc"),
    ("CUDA_ERROR_LAUNCH_FAILED", "hipErrorLaunchFailure"),
    ("cudaErrorLaunchFailure", "hipErrorLaunchFailure"),
    ("CUDA_ERROR_UNKNOWN", "hipErrorUnknown"),
    ("cudaErrorUnknown", "hipErrorUnknown"),
    ("CU_TR_ADDRESS_MODE_WRAP", "HIP_TR_ADDRESS_MODE_WRAP"),
    ("CU_TR_ADDRESS_MODE_CLAMP", "HIP_TR_ADDRESS_MODE_CLAMP"),
    ("CU_TR_ADDRESS_MODE_MIRROR", "HIP_TR_ADDRESS_MODE_MIRROR"),
    ("CU_TR_ADDRESS_MODE_BORDER", "HIP_TR_ADDRESS_MODE_BORDER"),
    ("CU_CUBEMAP_FACE_POSITIVE_X", "HIP_CUBEMAP_FACE_POSITIVE_X"),
    ("CU_CUBEMAP_FACE_NEGATIVE_X", "HIP_CUBEMAP_FACE_NEGATIVE_X"),
    ("CU_CUBEMAP_FACE_POSITIVE_Y", "HIP_CUBEMAP_FACE_POSITIVE_Y"),
    ("CU_CUBEMAP_FACE_NEGATIVE_Y", "HIP_CUBEMAP_FACE_NEGATIVE_Y"),
    ("CU_CUBEMAP_FACE_POSITIVE_Z", "HIP_CUBEMAP_FACE_POSITIVE_Z"),
    ("CU_CUBEMAP_FACE_NEGATIVE_Z", "HIP_CUBEMAP_FACE_NEGATIVE_Z"),
    ("CU_AD_FORMAT_UNSIGNED_INT8", "HIP_AD_FORMAT_UNSIGNED_INT8"),
    ("CU_AD_FORMAT_UNSIGNED_INT16", "HIP_AD_FORMAT_UNSIGNED_INT16"),
    ("CU_AD_FORMAT_UNSIGNED_INT32", "HIP_AD_FORMAT_UNSIGNED_INT32"),
    ("CU_AD_FORMAT_SIGNED_INT8", "HIP_AD_FORMAT_SIGNED_INT8"),
    ("CU_AD_FORMAT_SIGNED_INT16", "HIP_AD_FORMAT_SIGNED_INT16"),
    ("CU_AD_FORMAT_SIGNED_INT32", "HIP_AD_FORMAT_SIGNED_INT32"),
    ("CU_AD_FORMAT_HALF", "HIP_AD_FORMAT_HALF"),
    ("CU_AD_FORMAT_FLOAT", "HIP_AD_FORMAT_FLOAT"),
    ("CU_COMPUTEMODE_DEFAULT", "hipComputeModeDefault"),
    ("CU_COMPUTEMODE_EXCLUSIVE", "hipComputeModeExclusive"),
    ("CU_COMPUTEMODE_PROHIBITED", "hipComputeModeProhibited"),
    ("CU_COMPUTEMODE_EXCLUSIVE_PROCESS", "hipComputeModeExclusiveProcess"),
    ("CU_MEM_ADVISE_SET_READ_MOSTLY", "hipMemAdviseSetReadMostly"),
    ("CU_MEM_ADVISE_UNSET_READ_MOSTLY", "hipMemAdviseUnsetReadMostly"),
    ("CU_MEM_ADVISE_SET_PREFERRED_LOCATION", "hipMemAdviseSetPreferredLocation"),
    ("CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION", "hipMemAdviseUnsetPreferredLocation"),
    ("CU_MEM_ADVISE_SET_ACCESSED_BY", "hipMemAdviseSetAccessedBy"),
    ("CU_MEM_ADVISE_UNSET_ACCESSED_BY", "hipMemAdviseUnsetAccessedBy"),
    ("CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY", "hipMemRangeAttributeReadMostly"),
    ("CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION", "hipMemRangeAttributePreferredLocation"),
    ("CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY", "hipMemRangeAttributeAccessedBy"),
    ("CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION", "hipMemRangeAttributeLastPrefetchLocation"),
    ("CU_CTX_SCHED_AUTO", "HIP_CTX_SCHED_AUTO"),
    ("CU_CTX_SCHED_SPIN", "HIP_CTX_SCHED_SPIN"),
    ("CU_CTX_SCHED_YIELD", "HIP_CTX_SCHED_YIELD"),
    ("CU_CTX_SCHED_BLOCKING_SYNC", "HIP_CTX_SCHED_BLOCKING_SYNC"),
    ("CU_CTX_BLOCKING_SYNC", "HIP_CTX_BLOCKING_SYNC"),
    ("CU_CTX_SCHED_MASK", "HIP_CTX_SCHED_MASK"),
    ("CU_CTX_MAP_HOST", "HIP_CTX_MAP_HOST"),
    ("CU_CTX_LMEM_RESIZE_TO_MAX", "HIP_CTX_LMEM_RESIZE_TO_MAX"),
    ("CU_CTX_FLAGS_MASK", "HIP_CTX_FLAGS_MASK"),
    ("CU_LAUNCH_PARAM_BUFFER_POINTER", "HIP_LAUNCH_PARAM_BUFFER_POINTER"),
    ("CU_LAUNCH_PARAM_BUFFER_SIZE", "HIP_LAUNCH_PARAM_BUFFER_SIZE"),
    ("CU_LAUNCH_PARAM_END", "HIP_LAUNCH_PARAM_END"),
    ("CU_IPC_HANDLE_SIZE", "HIP_IPC_HANDLE_SIZE"),
    ("CU_MEMHOSTALLOC_DEVICEMAP", "HIP_MEMHOSTALLOC_DEVICEMAP"),
    ("CU_MEMHOSTALLOC_PORTABLE", "HIP_MEMHOSTALLOC_PORTABLE"),
    ("CU_MEMHOSTALLOC_WRITECOMBINED", "HIP_MEMHOSTALLOC_WRITECOMBINED"),
    ("CU_MEMHOSTREGISTER_DEVICEMAP", "HIP_MEMHOSTREGISTER_DEVICEMAP"),
    ("CU_MEMHOSTREGISTER_IOMEMORY", "HIP_MEMHOSTREGISTER_IOMEMORY"),
    ("CU_MEMHOSTREGISTER_PORTABLE", "HIP_MEMHOSTREGISTER_PORTABLE"),
    ("CU_PARAM_TR_DEFAULT", "HIP_PARAM_TR_DEFAULT"),
    ("CU_STREAM_LEGACY", "HIP_STREAM_LEGACY"),
    ("CU_STREAM_PER_THREAD", "HIP_STREAM_PER_THREAD"),
    ("CU_TRSA_OVERRIDE_FORMAT", "HIP_TRSA_OVERRIDE_FORMAT"),
    ("CU_TRSF_NORMALIZED_COORDINATES", "HIP_TRSF_NORMALIZED_COORDINATES"),
    ("CU_TRSF_READ_AS_INTEGER", "HIP_TRSF_READ_AS_INTEGER"),
    ("CU_TRSF_SRGB", "HIP_TRSF_SRGB"),
    ("CUDA_ARRAY3D_2DARRAY", "HIP_ARRAY3D_LAYERED"),
    ("CUDA_ARRAY3D_CUBEMAP", "HIP_ARRAY3D_CUBEMAP"),
    ("CUDA_ARRAY3D_DEPTH_TEXTURE", "HIP_ARRAY3D_DEPTH_TEXTURE"),
    ("CUDA_ARRAY3D_LAYERED", "HIP_ARRAY3D_LAYERED"),
    ("CUDA_ARRAY3D_SURFACE_LDST", "HIP_ARRAY3D_SURFACE_LDST"),
    ("CUDA_ARRAY3D_TEXTURE_GATHER", "HIP_ARRAY3D_TEXTURE_GATHER"),
    ("CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK", "hipDeviceAttributeMaxThreadsPerBlock"),
    ("CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X", "hipDeviceAttributeMaxBlockDimX"),
    ("CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y", "hipDeviceAttributeMaxBlockDimY"),
    ("CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z", "hipDeviceAttributeMaxBlockDimZ"),
    ("CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X", "hipDeviceAttributeMaxGridDimX"),
    ("CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y", "hipDeviceAttributeMaxGridDimY"),
    ("CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z", "hipDeviceAttributeMaxGridDimZ"),
    ("CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK", "hipDeviceAttributeMaxSharedMemoryPerBlock"),
    ("CU_DEVICE_ATTRIBUTE_SHARED_MEMORY_PER_BLOCK", "hipDeviceAttributeMaxSharedMemoryPerBlock"),
    ("CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY", "hipDeviceAttributeTotalConstantMemory"),
    ("CU_DEVICE_ATTRIBUTE_WARP_SIZE", "hipDeviceAttributeWarpSize"),
    ("CU_DEVICE_ATTRIBUTE_MAX_PITCH", "hipDeviceAttributeMaxPitch"),
    ("CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK", "hipDeviceAttributeMaxRegistersPerBlock"),
    ("CU_DEVICE_ATTRIBUTE_REGISTERS_PER_BLOCK", "hipDeviceAttributeMaxRegistersPerBlock"),
    ("CU_DEVICE_ATTRIBUTE_CLOCK_RATE", "hipDeviceAttributeClockRate"),
    ("CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT", "hipDeviceAttributeTextureAlignment"),
    ("CU_DEVICE_ATTRIBUTE_GPU_OVERLAP", "hipDeviceAttributeAsyncEngineCount"),
    ("CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT", "hipDeviceAttributeMultiprocessorCount"),
    ("CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT", "hipDeviceAttributeKernelExecTimeout"),
    ("CU_DEVICE_ATTRIBUTE_INTEGRATED", "hipDeviceAttributeIntegrated"),
    ("CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY", "hipDeviceAttributeCanMapHostMemory"),
    ("CU_DEVICE_ATTRIBUTE_COMPUTE_MODE", "hipDeviceAttributeComputeMode"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH", "hipDeviceAttributeMaxTexture1DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH", "hipDeviceAttributeMaxTexture2DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT", "hipDeviceAttributeMaxTexture2DHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH", "hipDeviceAttributeMaxTexture3DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT", "hipDeviceAttributeMaxTexture3DHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH", "hipDeviceAttributeMaxTexture3DDepth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH", "hipDeviceAttributeMaxTexture2DLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT", "hipDeviceAttributeMaxTexture2DLayeredHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS", "hipDeviceAttributeMaxTexture2DLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_WIDTH", "hipDeviceAttributeMaxTexture2DLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_HEIGHT", "hipDeviceAttributeMaxTexture2DLayeredHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_NUMSLICES", "hipDeviceAttributeMaxTexture2DLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_SURFACE_ALIGNMENT", "hipDeviceAttributeSurfaceAlignment"),
    ("CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS", "hipDeviceAttributeConcurrentKernels"),
    ("CU_DEVICE_ATTRIBUTE_ECC_ENABLED", "hipDeviceAttributeEccEnabled"),
    ("CU_DEVICE_ATTRIBUTE_PCI_BUS_ID", "hipDeviceAttributePciBusId"),
    ("CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID", "hipDeviceAttributePciDeviceId"),
    ("CU_DEVICE_ATTRIBUTE_TCC_DRIVER", "hipDeviceAttributeTccDriver"),
    ("CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE", "hipDeviceAttributeMemoryClockRate"),
    ("CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH", "hipDeviceAttributeMemoryBusWidth"),
    ("CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE", "hipDeviceAttributeL2CacheSize"),
    ("CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR", "hipDeviceAttributeMaxThreadsPerMultiProcessor"),
    ("CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT", "hipDeviceAttributeAsyncEngineCount"),
    ("CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING", "hipDeviceAttributeUnifiedAddressing"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH", "hipDeviceAttributeMaxTexture1DLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS", "hipDeviceAttributeMaxTexture1DLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_CAN_TEX2D_GATHER", "hipDeviceAttributeCanTex2DGather"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH", "hipDeviceAttributeMaxTexture2DGatherWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT", "hipDeviceAttributeMaxTexture2DGatherHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE", "hipDeviceAttributeMaxTexture3DWidthAlternate"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE", "hipDeviceAttributeMaxTexture3DHeightAlternate"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE", "hipDeviceAttributeMaxTexture3DDepthAlternate"),
    ("CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID", "hipDeviceAttributePciDomainId"),
    ("CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT", "hipDeviceAttributeTexturePitchAlignment"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH", "hipDeviceAttributeMaxTextureCubemapWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH", "hipDeviceAttributeMaxTextureCubemapLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS", "hipDeviceAttributeMaxTextureCubemapLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH", "hipDeviceAttributeMaxSurface1DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH", "hipDeviceAttributeMaxSurface2DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT", "hipDeviceAttributeMaxSurface2DHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH", "hipDeviceAttributeMaxSurface3DWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT", "hipDeviceAttributeMaxSurface3DHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH", "hipDeviceAttributeMaxSurface3DDepth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH", "hipDeviceAttributeMaxSurface1DLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS", "hipDeviceAttributeMaxSurface1DLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH", "hipDeviceAttributeMaxSurface2DLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT", "hipDeviceAttributeMaxSurface2DLayeredHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS", "hipDeviceAttributeMaxSurface2DLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH", "hipDeviceAttributeMaxSurfaceCubemapWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH", "hipDeviceAttributeMaxSurfaceCubemapLayeredWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS", "hipDeviceAttributeMaxSurfaceCubemapLayeredLayers"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH", "hipDeviceAttributeMaxTexture1DLinearWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH", "hipDeviceAttributeMaxTexture2DLinearWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT", "hipDeviceAttributeMaxTexture2DLinearHeight"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH", "hipDeviceAttributeMaxTexture2DLinearPitch"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH", "hipDeviceAttributeMaxTexture2DMipmappedWidth"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT", "hipDeviceAttributeMaxTexture2DMipmappedHeight"),
    ("CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR", "hipDeviceAttributeComputeCapabilityMajor"),
    ("CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR", "hipDeviceAttributeComputeCapabilityMinor"),
    ("CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH", "hipDeviceAttributeMaxTexture1DMipmappedWidth"),
    ("CU_DEVICE_ATTRIBUTE_STREAM_PRIORITIES_SUPPORTED", "hipDeviceAttributeStreamPrioritiesSupported"),
    ("CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED", "hipDeviceAttributeGlobalL1CacheSupported"),
    ("CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED", "hipDeviceAttributeLocalL1CacheSupported"),
    ("CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR", "hipDeviceAttributeMaxSharedMemoryPerMultiprocessor"),
    ("CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR", "hipDeviceAttributeMaxRegistersPerMultiprocessor"),
    ("CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY", "hipDeviceAttributeManagedMemory"),
    ("CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD", "hipDeviceAttributeIsMultiGpuBoard"),
    ("CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID", "hipDeviceAttributeMultiGpuBoardGroupId"),
    ("CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED", "hipDeviceAttributeHostNativeAtomicSupported"),
    ("CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO", "hipDeviceAttributeSingleToDoublePrecisionPerfRatio"),
    ("CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS", "hipDeviceAttributePageableMemoryAccess"),
    ("CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS", "hipDeviceAttributeConcurrentManagedAccess"),
    ("CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED", "hipDeviceAttributeComputePreemptionSupported"),
    ("CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM", "hipDeviceAttributeCanUseHostPointerForRegisteredMem"),
    ("CU_DEVICE_ATTRIBUTE_MAX", "hipDeviceAttributeMax"),
    ("CU_POINTER_ATTRIBUTE_CONTEXT", "hipPointerAttributeContext"),
    ("CU_POINTER_ATTRIBUTE_MEMORY_TYPE", "hipPointerAttributeMemoryType"),
    ("CU_POINTER_ATTRIBUTE_DEVICE_POINTER", "hipPointerAttributeDevicePointer"),
    ("CU_POINTER_ATTRIBUTE_HOST_POINTER", "hipPointerAttributeHostPointer"),
    ("CU_POINTER_ATTRIBUTE_P2P_TOKENS", "hipPointerAttributeP2pTokens"),
    ("CU_POINTER_ATTRIBUTE_SYNC_MEMOPS", "hipPointerAttributeSyncMemops"),
    ("CU_POINTER_ATTRIBUTE_BUFFER_ID", "hipPointerAttributeBufferId"),
    ("CU_POINTER_ATTRIBUTE_IS_MANAGED", "hipPointerAttributeIsManaged"),
    ("CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK", "hipFuncAttributeMaxThreadsPerBlocks"),
    ("CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES", "hipFuncAttributeSharedSizeBytes"),
    ("CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES", "hipFuncAttributeMaxDynamicSharedMemorySize"),
    ("CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES", "hipFuncAttributeConstSizeBytes"),
    ("CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES", "hipFuncAttributeLocalSizeBytes"),
    ("CU_FUNC_ATTRIBUTE_NUM_REGS", "hipFuncAttributeNumRegs"),
    ("CU_FUNC_ATTRIBUTE_PTX_VERSION", "hipFuncAttributePtxVersion"),
    ("CU_FUNC_ATTRIBUTE_BINARY_VERSION", "hipFuncAttributeBinaryVersion"),
    ("CU_FUNC_ATTRIBUTE_CACHE_MODE_CA", "hipFuncAttributeCacheModeCA"),
    ("CU_FUNC_ATTRIBUTE_MAX", "hipFuncAttributeMax"),
    ("CU_GRAPHICS_MAP_RESOURCE_FLAGS_NONE", "hipGraphicsMapFlagsNone"),
    ("CU_GRAPHICS_MAP_RESOURCE_FLAGS_READ_ONLY", "hipGraphicsMapFlagsReadOnly"),
    ("CU_GRAPHICS_MAP_RESOURCE_FLAGS_WRITE_DISCARD", "hipGraphicsMapFlagsWriteDiscard"),
    ("CU_GRAPHICS_REGISTER_FLAGS_NONE", "hipGraphicsRegisterFlagsNone"),
    ("CU_GRAPHICS_REGISTER_FLAGS_READ_ONLY", "hipGraphicsRegisterFlagsReadOnly"),
    ("CU_GRAPHICS_REGISTER_FLAGS_WRITE_DISCARD", "hipGraphicsRegisterFlagsWriteDiscard"),
    ("CU_GRAPHICS_REGISTER_FLAGS_SURFACE_LDST", "hipGraphicsRegisterFlagsSurfaceLoadStore"),
    ("CU_GRAPHICS_REGISTER_FLAGS_TEXTURE_GATHER", "hipGraphicsRegisterFlagsTextureGather"),
    ("CU_OCCUPANCY_DEFAULT", "hipOccupancyDefault"),
    ("CU_OCCUPANCY_DISABLE_CACHING_OVERRIDE", "hipOccupancyDisableCachingOverride"),
    ("CU_FUNC_CACHE_PREFER_NONE", "hipFuncCachePreferNone"),
    ("CU_FUNC_CACHE_PREFER_SHARED", "hipFuncCachePreferShared"),
    ("CU_FUNC_CACHE_PREFER_L1", "hipFuncCachePreferL1"),
    ("CU_FUNC_CACHE_PREFER_EQUAL", "hipFuncCachePreferEqual"),
    ("CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS", "hipIpcMemLazyEnablePeerAccess"),
    ("CUDA_IPC_HANDLE_SIZE", "HIP_IPC_HANDLE_SIZE"),
    ("CU_JIT_CACHE_OPTION_NONE", "hipJitCacheModeOptionNone"),
    ("CU_JIT_CACHE_OPTION_CG", "hipJitCacheModeOptionCG"),
    ("CU_JIT_CACHE_OPTION_CA", "hipJitCacheModeOptionCA"),
    ("CU_PREFER_PTX", "hipJitFallbackPreferPtx"),
    ("CU_PREFER_BINARY", "hipJitFallbackPreferBinary"),
    ("CU_JIT_MAX_REGISTERS", "hipJitOptionMaxRegisters"),
    ("CU_JIT_THREADS_PER_BLOCK", "hipJitOptionThreadsPerBlock"),
    ("CU_JIT_WALL_TIME", "hipJitOptionWallTime"),
    ("CU_JIT_INFO_LOG_BUFFER", "hipJitOptionInfoLogBuffer"),
    ("CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES", "hipJitOptionInfoLogBufferSizeBytes"),
    ("CU_JIT_ERROR_LOG_BUFFER", "hipJitOptionErrorLogBuffer"),
    ("CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES", "hipJitOptionErrorLogBufferSizeBytes"),
    ("CU_JIT_OPTIMIZATION_LEVEL", "hipJitOptionOptimizationLevel"),
    ("CU_JIT_TARGET_FROM_CUCONTEXT", "hipJitOptionTargetFromContext"),
    ("CU_JIT_TARGET", "hipJitOptionTarget"),
    ("CU_JIT_FALLBACK_STRATEGY", "hipJitOptionFallbackStrategy"),
    ("CU_JIT_GENERATE_DEBUG_INFO", "hipJitOptionGenerateDebugInfo"),
    ("CU_JIT_LOG_VERBOSE", "hipJitOptionLogVerbose"),
    ("CU_JIT_GENERATE_LINE_INFO", "hipJitOptionGenerateLineInfo"),
    ("CU_JIT_CACHE_MODE", "hipJitOptionCacheMode"),
    ("CU_JIT_NEW_SM3X_OPT", "hipJitOptionSm3xOpt"),
    ("CU_JIT_FAST_COMPILE", "hipJitOptionFastCompile"),
    ("CU_JIT_NUM_OPTIONS", "hipJitOptionNumOptions"),
    ("CU_TARGET_COMPUTE_10", "hipJitTargetCompute10"),
    ("CU_TARGET_COMPUTE_11", "hipJitTargetCompute11"),
    ("CU_TARGET_COMPUTE_12", "hipJitTargetCompute12"),
    ("CU_TARGET_COMPUTE_13", "hipJitTargetCompute13"),
    ("CU_TARGET_COMPUTE_20", "hipJitTargetCompute20"),
    ("CU_TARGET_COMPUTE_21", "hipJitTargetCompute21"),
    ("CU_TARGET_COMPUTE_30", "hipJitTargetCompute30"),
    ("CU_TARGET_COMPUTE_32", "hipJitTargetCompute32"),
    ("CU_TARGET_COMPUTE_35", "hipJitTargetCompute35"),
    ("CU_TARGET_COMPUTE_37", "hipJitTargetCompute37"),
    ("CU_TARGET_COMPUTE_50", "hipJitTargetCompute50"),
    ("CU_TARGET_COMPUTE_52", "hipJitTargetCompute52"),
    ("CU_TARGET_COMPUTE_53", "hipJitTargetCompute53"),
    ("CU_TARGET_COMPUTE_60", "hipJitTargetCompute60"),
    ("CU_TARGET_COMPUTE_61", "hipJitTargetCompute61"),
    ("CU_TARGET_COMPUTE_62", "hipJitTargetCompute62"),
    ("CU_JIT_INPUT_CUBIN", "hipJitInputTypeBin"),
    ("CU_JIT_INPUT_PTX", "hipJitInputTypePtx"),
    ("CU_JIT_INPUT_FATBINARY", "hipJitInputTypeFatBinary"),
    ("CU_JIT_INPUT_OBJECT", "hipJitInputTypeObject"),
    ("CU_JIT_INPUT_LIBRARY", "hipJitInputTypeLibrary"),
    ("CU_JIT_NUM_INPUT_TYPES", "hipJitInputTypeNumInputTypes"),
    ("CU_LIMIT_STACK_SIZE", "hipLimitStackSize"),
    ("CU_LIMIT_PRINTF_FIFO_SIZE", "hipLimitPrintfFifoSize"),
    ("CU_LIMIT_MALLOC_HEAP_SIZE", "hipLimitMallocHeapSize"),
    ("CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH", "hipLimitDevRuntimeSyncDepth"),
    ("CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT", "hipLimitDevRuntimePendingLaunchCount"),
    ("CU_MEM_ATTACH_GLOBAL", "hipMemAttachGlobal"),
    ("CU_MEM_ATTACH_HOST", "hipMemAttachHost"),
    ("CU_MEM_ATTACH_SINGLE", "hipMemAttachSingle"),
    ("CU_MEMORYTYPE_HOST", "hipMemTypeHost"),
    ("CU_MEMORYTYPE_DEVICE", "hipMemTypeDevice"),
    ("CU_MEMORYTYPE_ARRAY", "hipMemTypeArray"),
    ("CU_MEMORYTYPE_UNIFIED", "hipMemTypeUnified"),
    ("CU_RESOURCE_TYPE_ARRAY", "hipResourceTypeArray"),
    ("CU_RESOURCE_TYPE_MIPMAPPED_ARRAY", "hipResourceTypeMipmappedArray"),
    ("CU_RESOURCE_TYPE_LINEAR", "hipResourceTypeLinear"),
    ("CU_RESOURCE_TYPE_PITCH2D", "hipResourceTypePitch2D"),
    ("CU_RES_VIEW_FORMAT_NONE", "hipResViewFormatNone"),
    ("CU_RES_VIEW_FORMAT_UINT_1X8", "hipResViewFormatUnsignedChar1"),
    ("CU_RES_VIEW_FORMAT_UINT_2X8", "hipResViewFormatUnsignedChar2"),
    ("CU_RES_VIEW_FORMAT_UINT_4X8", "hipResViewFormatUnsignedChar4"),
    ("CU_RES_VIEW_FORMAT_SINT_1X8", "hipResViewFormatSignedChar1"),
    ("CU_RES_VIEW_FORMAT_SINT_2X8", "hipResViewFormatSignedChar2"),
    ("CU_RES_VIEW_FORMAT_SINT_4X8", "hipResViewFormatSignedChar4"),
    ("CU_RES_VIEW_FORMAT_UINT_1X16", "hipResViewFormatUnsignedShort1"),
    ("CU_RES_VIEW_FORMAT_UINT_2X16", "hipResViewFormatUnsignedShort2"),
    ("CU_RES_VIEW_FORMAT_UINT_4X16", "hipResViewFormatUnsignedShort4"),
    ("CU_RES_VIEW_FORMAT_SINT_1X16", "hipResViewFormatSignedShort1"),
    ("CU_RES_VIEW_FORMAT_SINT_2X16", "hipResViewFormatSignedShort2"),
    ("CU_RES_VIEW_FORMAT_SINT_4X16", "hipResViewFormatSignedShort4"),
    ("CU_RES_VIEW_FORMAT_UINT_1X32", "hipResViewFormatUnsignedInt1"),
    ("CU_RES_VIEW_FORMAT_UINT_2X32", "hipResViewFormatUnsignedInt2"),
    ("CU_RES_VIEW_FORMAT_UINT_4X32", "hipResViewFormatUnsignedInt4"),
    ("CU_RES_VIEW_FORMAT_SINT_1X32", "hipResViewFormatSignedInt1"),
    ("CU_RES_VIEW_FORMAT_SINT_2X32", "hipResViewFormatSignedInt2"),
    ("CU_RES_VIEW_FORMAT_SINT_4X32", "hipResViewFormatSignedInt4"),
    ("CU_RES_VIEW_FORMAT_FLOAT_1X16", "hipResViewFormatHalf1"),
    ("CU_RES_VIEW_FORMAT_FLOAT_2X16", "hipResViewFormatHalf2"),
    ("CU_RES_VIEW_FORMAT_FLOAT_4X16", "hipResViewFormatHalf4"),
    ("CU_RES_VIEW_FORMAT_FLOAT_1X32", "hipResViewFormatFloat1"),
    ("CU_RES_VIEW_FORMAT_FLOAT_2X32", "hipResViewFormatFloat2"),
    ("CU_RES_VIEW_FORMAT_FLOAT_4X32", "hipResViewFormatFloat4"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC1", "hipResViewFormatUnsignedBlockCompressed1"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC2", "hipResViewFormatUnsignedBlockCompressed2"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC3", "hipResViewFormatUnsignedBlockCompressed3"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC4", "hipResViewFormatUnsignedBlockCompressed4"),
    ("CU_RES_VIEW_FORMAT_SIGNED_BC4", "hipResViewFormatSignedBlockCompressed4"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC5", "hipResViewFormatUnsignedBlockCompressed5"),
    ("CU_RES_VIEW_FORMAT_SIGNED_BC5", "hipResViewFormatSignedBlockCompressed5"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC6H", "hipResViewFormatUnsignedBlockCompressed6H"),
    ("CU_RES_VIEW_FORMAT_SIGNED_BC6H", "hipResViewFormatSignedBlockCompressed6H"),
    ("CU_RES_VIEW_FORMAT_UNSIGNED_BC7", "hipResViewFormatUnsignedBlockCompressed7"),
    ("CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE", "hipSharedMemBankSizeDefault"),
    ("CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE", "hipSharedMemBankSizeFourByte"),
    ("CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE", "hipSharedMemBankSizeEightByte"),
    ("CU_STREAM_DEFAULT", "hipStreamDefault"),
    ("CU_STREAM_NON_BLOCKING", "hipStreamNonBlocking"),
    ("CU_STREAM_WAIT_VALUE_GEQ", "hipStreamWaitValueGeq"),
    ("CU_STREAM_WAIT_VALUE_EQ", "hipStreamWaitValueEq"),
    ("CU_STREAM_WAIT_VALUE_AND", "hipStreamWaitValueAnd"),
    ("CU_STREAM_WAIT_VALUE_FLUSH", "hipStreamWaitValueFlush"),
    ("CU_STREAM_WRITE_VALUE_DEFAULT", "hipStreamWriteValueDefault"),
    ("CU_STREAM_WRITE_VALUE_NO_MEMORY_BARRIER", "hipStreamWriteValueNoMemoryBarrier"),
    ("CU_STREAM_MEM_OP_WAIT_VALUE_32", "hipStreamBatchMemOpWaitValue32"),
    ("CU_STREAM_MEM_OP_WRITE_VALUE_32", "hipStreamBatchMemOpWriteValue32"),
    ("CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES", "hipStreamBatchMemOpFlushRemoteWrites"),
    ("cuGetErrorName", "hipGetErrorName"),
    ("cuGetErrorString", "hipDrvGetErrorString"),
    ("cuInit", "hipInit"),
    ("cuDriverGetVersion", "hipDriverGetVersion"),
    ("cuCtxCreate", "hipCtxCreate"),
    ("cuCtxCreate_v2", "hipCtxCreate"),
    ("cuCtxDestroy", "hipCtxDestroy"),
    ("cuCtxDestroy_v2", "hipCtxDestroy"),
    ("cuCtxGetApiVersion", "hipCtxGetApiVersion"),
    ("cuCtxGetCacheConfig", "hipCtxGetCacheConfig"),
    ("cuCtxGetCurrent", "hipCtxGetCurrent"),
    ("cuCtxGetDevice", "hipCtxGetDevice"),
    ("cuCtxGetFlags", "hipCtxGetFlags"),
    ("cuDeviceGetUuid", "hipDeviceGetUuid"),
    ("cuCtxGetLimit", "hipCtxGetLimit"),
    ("cuCtxGetSharedMemConfig", "hipCtxGetSharedMemConfig"),
    ("cuCtxGetStreamPriorityRange", "hipCtxGetStreamPriorityRange"),
    ("cuCtxPopCurrent_v2", "hipCtxPopCurrent"),
    ("cuCtxPushCurrent_v2", "hipCtxPushCurrent"),
    ("cuCtxSetCacheConfig", "hipCtxSetCacheConfig"),
    ("cuCtxSetCurrent", "hipCtxSetCurrent"),
    ("cuCtxSetLimit", "hipCtxSetLimit"),
    ("cuCtxSetSharedMemConfig", "hipCtxSetSharedMemConfig"),
    ("cuCtxSynchronize", "hipCtxSynchronize"),
    ("cuCtxAttach", "hipCtxAttach"),
    ("cuCtxDetach", "hipCtxDetach"),
    ("cuCtxEnablePeerAccess", "hipCtxEnablePeerAccess"),
    ("cuCtxDisablePeerAccess", "hipCtxDisablePeerAccess"),
    ("cuDeviceCanAccessPeer", "hipDeviceCanAccessPeer"),
    ("cuDeviceGetP2PAttribute", "hipDeviceGetP2PAttribute"),
    ("cuDevicePrimaryCtxGetState", "hipDevicePrimaryCtxGetState"),
    ("cuDevicePrimaryCtxRelease", "hipDevicePrimaryCtxRelease"),
    ("cuDevicePrimaryCtxReset", "hipDevicePrimaryCtxReset"),
    ("cuDevicePrimaryCtxRetain", "hipDevicePrimaryCtxRetain"),
    ("cuDevicePrimaryCtxSetFlags", "hipDevicePrimaryCtxSetFlags"),
    ("cuDeviceGet", "hipDeviceGet"),
    ("cuDeviceGetName", "hipDeviceGetName"),
    ("cuDeviceGetCount", "hipGetDeviceCount"),
    ("cuDeviceGetAttribute", "hipDeviceGetAttribute"),
    ("cuDeviceGetPCIBusId", "hipDeviceGetPCIBusId"),
    ("cuDeviceGetByPCIBusId", "hipDeviceGetByPCIBusId"),
    ("cuDeviceTotalMem_v2", "hipDeviceTotalMem"),
    ("cuDeviceComputeCapability", "hipDeviceComputeCapability"),
    ("cuDeviceGetProperties", "hipGetDeviceProperties"),
    ("cuLinkAddData", "hipLinkAddData"),
    ("cuLinkAddFile", "hipLinkAddFile"),
    ("cuLinkComplete", "hipLinkComplete"),
    ("cuLinkCreate", "hipLinkCreate"),
    ("cuLinkDestroy", "hipLinkDestroy"),
    ("cuModuleGetFunction", "hipModuleGetFunction"),
    ("cuModuleGetGlobal", "hipModuleGetGlobal"),
    ("cuModuleGetGlobal_v2", "hipModuleGetGlobal"),
    ("cuModuleGetSurfRef", "hipModuleGetSurfRef"),
    ("cuModuleGetTexRef", "hipModuleGetTexRef"),
    ("cuModuleLoad", "hipModuleLoad"),
    ("cuModuleLoadData", "hipModuleLoadData"),
    ("cuModuleLoadDataEx", "hipModuleLoadDataEx"),
    ("cuModuleLoadFatBinary", "hipModuleLoadFatBinary"),
    ("cuModuleUnload", "hipModuleUnload"),
    ("CU_DEVICE_P2P_ATTRIBUTE_PERFORMANCE_RANK", "hipDeviceP2PAttributePerformanceRank"),
    ("CU_DEVICE_P2P_ATTRIBUTE_ACCESS_SUPPORTED", "hipDeviceP2PAttributeAccessSupported"),
    ("CU_DEVICE_P2P_ATTRIBUTE_NATIVE_ATOMIC_SUPPORTED", "hipDeviceP2PAttributeNativeAtomicSupported"),
    ("CU_EVENT_DEFAULT", "hipEventDefault"),
    ("CU_EVENT_BLOCKING_SYNC", "hipEventBlockingSync"),
    ("CU_EVENT_DISABLE_TIMING", "hipEventDisableTiming"),
    ("CU_EVENT_INTERPROCESS", "hipEventInterprocess"),
    ("cuEventCreate", "hipEventCreate"),
    ("cuEventDestroy", "hipEventDestroy"),
    ("cuEventDestroy_v2", "hipEventDestroy"),
    ("cuEventElapsedTime", "hipEventElapsedTime"),
    ("cuEventQuery", "hipEventQuery"),
    ("cuEventRecord", "hipEventRecord"),
    ("cuEventSynchronize", "hipEventSynchronize"),
    ("cuFuncSetAttribute", "hipFuncSetAttribute"),
    ("cuFuncGetAttribute", "hipFuncGetAttribute"),
    ("cuFuncSetCacheConfig", "hipFuncSetCacheConfig"),
    ("cuFuncSetSharedMemConfig", "hipFuncSetSharedMemConfig"),
    ("cuLaunchKernel", "hipModuleLaunchKernel"),
    ("cuLaunchCooperativeKernel", "hipModuleLaunchCooperativeKernel"),
    ("cuFuncSetBlockShape", "hipFuncSetBlockShape"),
    ("cuFuncSetSharedSize", "hipFuncSetSharedSize"),
    ("cuLaunch", "hipLaunch"),
    ("cuLaunchGrid", "hipLaunchGrid"),
    ("cuLaunchGridAsync", "hipLaunchGridAsync"),
    ("cuParamSetf", "hipParamSetf"),
    ("cuParamSeti", "hipParamSeti"),
    ("cuParamSetSize", "hipParamSetSize"),
    ("cuParamSetv", "hipParamSetv"),
    ("cuOccupancyMaxActiveBlocksPerMultiprocessor", "hipModuleOccupancyMaxActiveBlocksPerMultiprocessor"),
    ("cuOccupancyMaxActiveBlocksPerMultiprocessorWithFlags", "hipModuleOccupancyMaxActiveBlocksPerMultiprocessorWithFlags"),
    ("cuOccupancyMaxPotentialBlockSize", "hipModuleOccupancyMaxPotentialBlockSize"),
    ("cuOccupancyMaxPotentialBlockSizeWithFlags", "hipModuleOccupancyMaxPotentialBlockSizeWithFlags"),
    ("cuStreamAddCallback", "hipStreamAddCallback"),
    ("cuStreamAttachMemAsync", "hipStreamAttachMemAsync"),
    ("cuStreamCreate", "hipStreamCreate__"),
    ("cuStreamCreateWithPriority", "hipStreamCreateWithPriority"),
    ("cuStreamDestroy", "hipStreamDestroy"),
    ("cuStreamDestroy_v2", "hipStreamDestroy"),
    ("cuStreamGetFlags", "hipStreamGetFlags"),
    ("cuStreamGetPriority", "hipStreamGetPriority"),
    ("cuStreamQuery", "hipStreamQuery"),
    ("cuStreamSynchronize", "hipStreamSynchronize"),
    ("cuStreamWaitEvent", "hipStreamWaitEvent"),
    ("cuStreamWaitValue32", "hipStreamWaitValue32"),
    ("cuStreamWriteValue32", "hipStreamWriteValue32"),
    ("cuStreamBatchMemOp", "hipStreamBatchMemOp"),
    ("cuArray3DCreate", "hipArray3DCreate"),
    ("cuArray3DGetDescriptor", "hipArray3DGetDescriptor"),
    ("cuArrayCreate", "hipArrayCreate"),
    ("cuArrayDestroy", "hipArrayDestroy"),
    ("cuArrayGetDescriptor", "hipArrayGetDescriptor"),
    ("cuIpcCloseMemHandle", "hipIpcCloseMemHandle"),
    ("cuIpcGetEventHandle", "hipIpcGetEventHandle"),
    ("cuIpcGetMemHandle", "hipIpcGetMemHandle"),
    ("cuIpcOpenEventHandle", "hipIpcOpenEventHandle"),
    ("cuIpcOpenMemHandle", "hipIpcOpenMemHandle"),
    ("cuMemAlloc_v2", "hipMalloc"),
    ("cuMemAllocHost", "hipMemAllocHost"),
    ("cuMemAllocManaged", "hipMemAllocManaged"),
    ("cuMemAllocPitch", "hipMemAllocPitch__"),
    ("cuMemcpy", "hipMemcpy__"),
    ("cuMemcpy2D", "hipMemcpy2D__"),
    ("cuMemcpy2DAsync", "hipMemcpy2DAsync__"),
    ("cuMemcpy2DUnaligned", "hipMemcpy2DUnaligned"),
    ("cuMemcpy3D", "hipMemcpy3D__"),
    ("cuMemcpy3DAsync", "hipMemcpy3DAsync__"),
    ("cuMemcpy3DPeer", "hipMemcpy3DPeer__"),
    ("cuMemcpy3DPeerAsync", "hipMemcpy3DPeerAsync__"),
    ("cuMemcpyAsync", "hipMemcpyAsync__"),
    ("cuMemcpyAtoA", "hipMemcpyAtoA"),
    ("cuMemcpyAtoD", "hipMemcpyAtoD"),
    ("cuMemcpyAtoH", "hipMemcpyAtoH"),
    ("cuMemcpyAtoHAsync", "hipMemcpyAtoHAsync"),
    ("cuMemcpyDtoA", "hipMemcpyDtoA"),
    ("cuMemcpyDtoD_v2", "hipMemcpyDtoD"),
    ("cuMemcpyDtoDAsync_v2", "hipMemcpyDtoDAsync"),
    ("cuMemcpyDtoH_v2", "hipMemcpyDtoH"),
    ("cuMemcpyDtoHAsync_v2", "hipMemcpyDtoHAsync"),
    ("cuMemcpyHtoA", "hipMemcpyHtoA"),
    ("cuMemcpyHtoAAsync", "hipMemcpyHtoAAsync"),
    ("cuMemcpyHtoD_v2", "hipMemcpyHtoD"),
    ("cuMemcpyHtoDAsync_v2", "hipMemcpyHtoDAsync"),
    ("cuMemcpyPeerAsync", "hipMemcpyPeerAsync__"),
    ("cuMemcpyPeer", "hipMemcpyPeer__"),
    ("cuMemFree", "hipFree"),
    ("cuMemFree_v2", "hipFree"),
    ("cuMemFreeHost", "hipHostFree"),
    ("cuMemGetAddressRange", "hipMemGetAddressRange"),
    ("cuMemGetInfo_v2", "hipMemGetInfo"),
    ("cuMemHostAlloc", "hipHostMalloc"),
    ("cuMemHostGetDevicePointer", "hipMemHostGetDevicePointer"),
    ("cuMemHostGetFlags", "hipMemHostGetFlags"),
    ("cuMemHostRegister_v2", "hipHostRegister"),
    ("cuMemHostUnregister", "hipHostUnregister"),
    ("cuMemsetD16_v2", "hipMemsetD16"),
    ("cuMemsetD16Async", "hipMemsetD16Async"),
    ("cuMemsetD2D16_v2", "hipMemsetD2D16"),
    ("cuMemsetD2D16Async", "hipMemsetD2D16Async"),
    ("cuMemsetD2D32_v2", "hipMemsetD2D32"),
    ("cuMemsetD2D32Async", "hipMemsetD2D32Async"),
    ("cuMemsetD2D8_v2", "hipMemsetD2D8"),
    ("cuMemsetD2D8Async", "hipMemsetD2D8Async"),
    ("cuMemsetD32_v2", "hipMemset"),
    ("cuMemsetD32Async", "hipMemsetAsync"),
    ("cuMemsetD8_v2", "hipMemsetD8"),
    ("cuMemsetD8Async", "hipMemsetD8Async"),
    ("cuMipmappedArrayCreate", "hipMipmappedArrayCreate"),
    ("cuMipmappedArrayDestroy", "hipMipmappedArrayDestroy"),
    ("cuMipmappedArrayGetLevel", "hipMipmappedArrayGetLevel"),
    ("cuMemPrefetchAsync", "hipMemPrefetchAsync__"),
    ("cuMemAdvise", "hipMemAdvise"),
    ("cuMemRangeGetAttribute", "hipMemRangeGetAttribute"),
    ("cuMemRangeGetAttributes", "hipMemRangeGetAttributes"),
    ("cuPointerGetAttribute", "hipPointerGetAttribute"),
    ("cuMemGetAddressRange_v2", "hipMemGetAddressRange"),
    ("cuPointerGetAttributes", "hipPointerGetAttributes"),
    ("cuPointerSetAttribute", "hipPointerSetAttribute"),
    ("CU_TR_FILTER_MODE_POINT", "hipFilterModePoint"),
    ("CU_TR_FILTER_MODE_LINEAR", "hipFilterModeLinear"),
    ("cuTexRefGetAddress", "hipTexRefGetAddress"),
    ("cuTexRefGetAddressMode", "hipTexRefGetAddressMode"),
    ("cuTexRefGetArray", "hipTexRefGetArray"),
    ("cuTexRefGetBorderColor", "hipTexRefGetBorderColor"),
    ("cuTexRefGetFilterMode", "hipTexRefGetFilterMode"),
    ("cuTexRefGetFlags", "hipTexRefGetFlags"),
    ("cuTexRefGetFormat", "hipTexRefGetFormat"),
    ("cuTexRefGetMaxAnisotropy", "hipTexRefGetMaxAnisotropy"),
    ("cuTexRefGetMipmapFilterMode", "hipTexRefGetMipmapFilterMode"),
    ("cuTexRefGetMipmapLevelBias", "hipTexRefGetMipmapLevelBias"),
    ("cuTexRefGetMipmapLevelClamp", "hipTexRefGetMipmapLevelClamp"),
    ("cuTexRefGetMipmappedArray", "hipTexRefGetMipmappedArray"),
    ("cuTexRefSetAddress", "hipTexRefSetAddress"),
    ("cuTexRefSetAddress2D", "hipTexRefSetAddress2D"),
    ("cuTexRefSetAddressMode", "hipTexRefSetAddressMode"),
    ("cuTexRefSetArray", "hipTexRefSetArray"),
    ("cuTexRefSetBorderColor", "hipTexRefSetBorderColor"),
    ("cuTexRefSetFilterMode", "hipTexRefSetFilterMode"),
    ("cuTexRefSetFlags", "hipTexRefSetFlags"),
    ("cuTexRefSetFormat", "hipTexRefSetFormat"),
    ("cuTexRefSetMaxAnisotropy", "hipTexRefSetMaxAnisotropy"),
    ("cuTexRefSetMipmapFilterMode", "hipTexRefSetMipmapFilterMode"),
    ("cuTexRefSetMipmapLevelBias", "hipTexRefSetMipmapLevelBias"),
    ("cuTexRefSetMipmapLevelClamp", "hipTexRefSetMipmapLevelClamp"),
    ("cuTexRefSetMipmappedArray", "hipTexRefSetMipmappedArray"),
    ("cuTexRefCreate", "hipTexRefCreate"),
    ("cuTexRefDestroy", "hipTexRefDestroy"),
    ("cuSurfRefGetArray", "hipSurfRefGetArray"),
    ("cuSurfRefSetArray", "hipSurfRefSetArray"),
    ("cuTexObjectCreate", "hipTexObjectCreate"),
    ("cuTexObjectDestroy", "hipTexObjectDestroy"),
    ("cuTexObjectGetResourceDesc", "hipTexObjectGetResourceDesc"),
    ("cuTexObjectGetResourceViewDesc", "hipTexObjectGetResourceViewDesc"),
    ("cuTexObjectGetTextureDesc", "hipTexObjectGetTextureDesc"),
    ("cuSurfObjectCreate", "hipSurfObjectCreate"),
    ("cuSurfObjectDestroy", "hipSurfObjectDestroy"),
    ("cuSurfObjectGetResourceDesc", "hipSurfObjectGetResourceDesc"),
    ("cuGraphicsMapResources", "hipGraphicsMapResources"),
    ("cuGraphicsResourceGetMappedMipmappedArray", "hipGraphicsResourceGetMappedMipmappedArray"),
    ("cuGraphicsResourceGetMappedPointer", "hipGraphicsResourceGetMappedPointer"),
    ("cuGraphicsResourceSetMapFlags", "hipGraphicsResourceSetMapFlags"),
    ("cuGraphicsSubResourceGetMappedArray", "hipGraphicsSubResourceGetMappedArray"),
    ("cuGraphicsUnmapResources", "hipGraphicsUnmapResources"),
    ("cuGraphicsUnregisterResource", "hipGraphicsUnregisterResource"),
    ("cuProfilerInitialize", "hipProfilerInitialize"),
    ("cuProfilerStart", "hipProfilerStart"),
    ("cuProfilerStop", "hipProfilerStop"),
    ("CU_GL_DEVICE_LIST_ALL", "HIP_GL_DEVICE_LIST_ALL"),
    ("CU_GL_DEVICE_LIST_CURRENT_FRAME", "HIP_GL_DEVICE_LIST_CURRENT_FRAME"),
    ("CU_GL_DEVICE_LIST_NEXT_FRAME", "HIP_GL_DEVICE_LIST_NEXT_FRAME"),
    ("cuGLGetDevices", "hipGLGetDevices"),
    ("cuGraphicsGLRegisterBuffer", "hipGraphicsGLRegisterBuffer"),
    ("cuGraphicsGLRegisterImage", "hipGraphicsGLRegisterImage"),
    ("cuWGLGetDevice", "hipWGLGetDevice"),
    ("CU_GL_MAP_RESOURCE_FLAGS_NONE", "HIP_GL_MAP_RESOURCE_FLAGS_NONE"),
    ("CU_GL_MAP_RESOURCE_FLAGS_READ_ONLY", "HIP_GL_MAP_RESOURCE_FLAGS_READ_ONLY"),
    ("CU_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD", "HIP_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD"),
    ("cuGLCtxCreate", "hipGLCtxCreate"),
    ("cuGLInit", "hipGLInit"),
    ("cuGLMapBufferObject", "hipGLMapBufferObject"),
    ("cuGLMapBufferObjectAsync", "hipGLMapBufferObjectAsync"),
    ("cuGLRegisterBufferObject", "hipGLRegisterBufferObject"),
    ("cuGLSetBufferObjectMapFlags", "hipGLSetBufferObjectMapFlags"),
    ("cuGLUnmapBufferObject", "hipGLUnmapBufferObject"),
    ("cuGLUnmapBufferObjectAsync", "hipGLUnmapBufferObjectAsync"),
    ("cuGLUnregisterBufferObject", "hipGLUnregisterBufferObject"),
    ("CU_D3D9_DEVICE_LIST_ALL", "HIP_D3D9_DEVICE_LIST_ALL"),
    ("CU_D3D9_DEVICE_LIST_CURRENT_FRAME", "HIP_D3D9_DEVICE_LIST_CURRENT_FRAME"),
    ("CU_D3D9_DEVICE_LIST_NEXT_FRAME", "HIP_D3D9_DEVICE_LIST_NEXT_FRAME"),
    ("cuD3D9CtxCreate", "hipD3D9CtxCreate"),
    ("cuD3D9CtxCreateOnDevice", "hipD3D9CtxCreateOnDevice"),
    ("cuD3D9GetDevice", "hipD3D9GetDevice"),
    ("cuD3D9GetDevices", "hipD3D9GetDevices"),
    ("cuD3D9GetDirect3DDevice", "hipD3D9GetDirect3DDevice"),
    ("cuGraphicsD3D9RegisterResource", "hipGraphicsD3D9RegisterResource"),
    ("CU_D3D9_MAPRESOURCE_FLAGS_NONE", "HIP_D3D9_MAPRESOURCE_FLAGS_NONE"),
    ("CU_D3D9_MAPRESOURCE_FLAGS_READONLY", "HIP_D3D9_MAPRESOURCE_FLAGS_READONLY"),
    ("CU_D3D9_MAPRESOURCE_FLAGS_WRITEDISCARD", "HIP_D3D9_MAPRESOURCE_FLAGS_WRITEDISCARD"),
    ("CU_D3D9_REGISTER_FLAGS_NONE", "HIP_D3D9_REGISTER_FLAGS_NONE"),
    ("CU_D3D9_REGISTER_FLAGS_ARRAY", "HIP_D3D9_REGISTER_FLAGS_ARRAY"),
    ("cuD3D9MapResources", "hipD3D9MapResources"),
    ("cuD3D9RegisterResource", "hipD3D9RegisterResource"),
    ("cuD3D9ResourceGetMappedArray", "hipD3D9ResourceGetMappedArray"),
    ("cuD3D9ResourceGetMappedPitch", "hipD3D9ResourceGetMappedPitch"),
    ("cuD3D9ResourceGetMappedPointer", "hipD3D9ResourceGetMappedPointer"),
    ("cuD3D9ResourceGetMappedSize", "hipD3D9ResourceGetMappedSize"),
    ("cuD3D9ResourceGetSurfaceDimensions", "hipD3D9ResourceGetSurfaceDimensions"),
    ("cuD3D9ResourceSetMapFlags", "hipD3D9ResourceSetMapFlags"),
    ("cuD3D9UnmapResources", "hipD3D9UnmapResources"),
    ("cuD3D9UnregisterResource", "hipD3D9UnregisterResource"),
    ("CU_D3D10_DEVICE_LIST_ALL", "HIP_D3D10_DEVICE_LIST_ALL"),
    ("CU_D3D10_DEVICE_LIST_CURRENT_FRAME", "HIP_D3D10_DEVICE_LIST_CURRENT_FRAME"),
    ("CU_D3D10_DEVICE_LIST_NEXT_FRAME", "HIP_D3D10_DEVICE_LIST_NEXT_FRAME"),
    ("cuD3D10GetDevice", "hipD3D10GetDevice"),
    ("cuD3D10GetDevices", "hipD3D10GetDevices"),
    ("cuGraphicsD3D10RegisterResource", "hipGraphicsD3D10RegisterResource"),
    ("CU_D3D10_MAPRESOURCE_FLAGS_NONE", "HIP_D3D10_MAPRESOURCE_FLAGS_NONE"),
    ("CU_D3D10_MAPRESOURCE_FLAGS_READONLY", "HIP_D3D10_MAPRESOURCE_FLAGS_READONLY"),
    ("CU_D3D10_MAPRESOURCE_FLAGS_WRITEDISCARD", "HIP_D3D10_MAPRESOURCE_FLAGS_WRITEDISCARD"),
    ("CU_D3D10_REGISTER_FLAGS_NONE", "HIP_D3D10_REGISTER_FLAGS_NONE"),
    ("CU_D3D10_REGISTER_FLAGS_ARRAY", "HIP_D3D10_REGISTER_FLAGS_ARRAY"),
    ("cuD3D10CtxCreate", "hipD3D10CtxCreate"),
    ("cuD3D10CtxCreateOnDevice", "hipD3D10CtxCreateOnDevice"),
    ("cuD3D10GetDirect3DDevice", "hipD3D10GetDirect3DDevice"),
    ("cuD3D10MapResources", "hipD3D10MapResources"),
    ("cuD3D10RegisterResource", "hipD3D10RegisterResource"),
    ("cuD3D10ResourceGetMappedArray", "hipD3D10ResourceGetMappedArray"),
    ("cuD3D10ResourceGetMappedPitch", "hipD3D10ResourceGetMappedPitch"),
    ("cuD3D10ResourceGetMappedPointer", "hipD3D10ResourceGetMappedPointer"),
    ("cuD3D10ResourceGetMappedSize", "hipD3D10ResourceGetMappedSize"),
    ("cuD3D10ResourceGetSurfaceDimensions", "hipD3D10ResourceGetSurfaceDimensions"),
    ("cuD310ResourceSetMapFlags", "hipD3D10ResourceSetMapFlags"),
    ("cuD3D10UnmapResources", "hipD3D10UnmapResources"),
    ("cuD3D10UnregisterResource", "hipD3D10UnregisterResource"),
    ("CU_D3D11_DEVICE_LIST_ALL", "HIP_D3D11_DEVICE_LIST_ALL"),
    ("CU_D3D11_DEVICE_LIST_CURRENT_FRAME", "HIP_D3D11_DEVICE_LIST_CURRENT_FRAME"),
    ("CU_D3D11_DEVICE_LIST_NEXT_FRAME", "HIP_D3D11_DEVICE_LIST_NEXT_FRAME"),
    ("cuD3D11GetDevice", "hipD3D11GetDevice"),
    ("cuD3D11GetDevices", "hipD3D11GetDevices"),
    ("cuGraphicsD3D11RegisterResource", "hipGraphicsD3D11RegisterResource"),
    ("cuD3D11CtxCreate", "hipD3D11CtxCreate"),
    ("cuD3D11CtxCreateOnDevice", "hipD3D11CtxCreateOnDevice"),
    ("cuD3D11GetDirect3DDevice", "hipD3D11GetDirect3DDevice"),
    ("cuGraphicsVDPAURegisterOutputSurface", "hipGraphicsVDPAURegisterOutputSurface"),
    ("cuGraphicsVDPAURegisterVideoSurface", "hipGraphicsVDPAURegisterVideoSurface"),
    ("cuVDPAUGetDevice", "hipVDPAUGetDevice"),
    ("cuVDPAUCtxCreate", "hipVDPAUCtxCreate"),
    ("cuEGLStreamConsumerAcquireFrame", "hipEGLStreamConsumerAcquireFrame"),
    ("cuEGLStreamConsumerConnect", "hipEGLStreamConsumerConnect"),
    ("cuEGLStreamConsumerConnectWithFlags", "hipEGLStreamConsumerConnectWithFlags"),
    ("cuEGLStreamConsumerDisconnect", "hipEGLStreamConsumerDisconnect"),
    ("cuEGLStreamConsumerReleaseFrame", "hipEGLStreamConsumerReleaseFrame"),
    ("cuEGLStreamProducerConnect", "hipEGLStreamProducerConnect"),
    ("cuEGLStreamProducerDisconnect", "hipEGLStreamProducerDisconnect"),
    ("cuEGLStreamProducerPresentFrame", "hipEGLStreamProducerPresentFrame"),
    ("cuEGLStreamProducerReturnFrame", "hipEGLStreamProducerReturnFrame"),
    ("cuGraphicsEGLRegisterImage", "hipGraphicsEGLRegisterImage"),
    ("cuGraphicsResourceGetMappedEglFrame", "hipGraphicsResourceGetMappedEglFrame"),
    ("cudaDataType_t", "hipDataType"),
    ("cudaDataType", "hipDataType"),
    ("CUDA_R_32F", "HIP_R_32F"),
    ("CUDA_R_64F", "HIP_R_64F"),
    ("CUDA_R_16F", "HIP_R_16F"),
    ("CUDA_R_8I", "HIP_R_8I"),
    ("CUDA_C_32F", "HIP_C_32F"),
    ("CUDA_C_64F", "HIP_C_64F"),
    ("CUDA_C_16F", "HIP_C_16F"),
    ("CUDA_C_8I", "HIP_C_8I"),
    ("CUDA_R_8U", "HIP_R_8U"),
    ("CUDA_C_8U", "HIP_C_8U"),
    ("CUDA_R_32I", "HIP_R_32I"),
    ("CUDA_C_32I", "HIP_C_32I"),
    ("CUDA_R_32U", "HIP_R_32U"),
    ("CUDA_C_32U", "HIP_C_32U"),
    ("CUDA_R_16BF", "HIP_R_16BF"),
    ("CUDA_C_16BF", "HIP_C_16BF"),
    ("CUDA_R_4I", "HIP_R_4I"),
    ("CUDA_C_4I", "HIP_C_4I"),
    ("CUDA_R_4U", "HIP_R_4U"),
    ("CUDA_C_4U", "HIP_C_4U"),
    ("CUDA_R_16I", "HIP_R_16I"),
    ("CUDA_C_16I", "HIP_C_16I"),
    ("CUDA_R_16U", "HIP_R_16U"),
    ("CUDA_C_16U", "HIP_C_16U"),
    ("CUDA_R_64I", "HIP_R_64I"),
    ("CUDA_C_64I", "HIP_C_64I"),
    ("CUDA_R_64U", "HIP_R_64U"),
    ("CUDA_C_64U", "HIP_C_64U"),
    ("CUDA_R_8F_E4M3", "HIP_R_8F_E4M3"),
    ("CUDA_R_8F_E5M2", "HIP_R_8F_E5M2"),
    ("MAJOR_VERSION", "hipLibraryMajorVersion"),
    ("MINOR_VERSION", "hipLibraryMinorVersion"),
    ("PATCH_LEVEL", "hipLibraryPatchVersion"),
    ("cudaMemAllocationHandleType", "hipMemAllocationHandleType"),
    ("cudaMemAllocationType", "hipMemAllocationType"),
    ("cudaMemLocationType", "hipMemLocationType"),
    ("cudaMemAttachGlobal", "hipMemAttachGlobal"),
    ("cudaMemAttachHost", "hipMemAttachHost"),
    ("cudaMemAttachSingle", "hipMemAttachSingle"),
    ("cudaOccupancyDefault", "hipOccupancyDefault"),
    ("cudaOccupancyDisableCachingOverride", "hipOccupancyDisableCachingOverride"),
    ("cudaGetLastError", "hipGetLastError"),
    ("cudaPeekAtLastError", "hipPeekAtLastError"),
    ("cudaGetErrorName", "hipGetErrorName"),
    ("cudaGetErrorString", "hipGetErrorString"),
    ("cudaMemcpy3DParms", "hipMemcpy3DParms"),
    ("cudaMemcpy3DPeerParms", "hipMemcpy3DPeerParms"),
    ("cudaMemcpy", "hipMemcpy"),
    ("cudaMemcpyToArray", "hipMemcpyToArray"),
    ("cudaMemcpyToSymbol", "hipMemcpyToSymbol"),
    ("cudaMemcpyToSymbolAsync", "hipMemcpyToSymbolAsync"),
    ("cudaMemcpyAsync", "hipMemcpyAsync"),
    ("cudaMemcpy2D", "hipMemcpy2D"),
    ("cudaMemcpy2DAsync", "hipMemcpy2DAsync"),
    ("cudaMemcpy2DToArray", "hipMemcpy2DToArray"),
    ("cudaMemcpy2DArrayToArray", "hipMemcpy2DArrayToArray"),
    ("cudaMemcpy2DFromArray", "hipMemcpy2DFromArray"),
    ("cudaMemcpy2DFromArrayAsync", "hipMemcpy2DFromArrayAsync"),
    ("cudaMemcpy2DToArrayAsync", "hipMemcpy2DToArrayAsync"),
    ("cudaMemcpy3D", "hipMemcpy3D"),
    ("cudaMemcpy3DAsync", "hipMemcpy3DAsync"),
    ("cudaMemcpy3DPeer", "hipMemcpy3DPeer"),
    ("cudaMemcpy3DPeerAsync", "hipMemcpy3DPeerAsync"),
    ("cudaMemcpyArrayToArray", "hipMemcpyArrayToArray"),
    ("cudaMemcpyFromArrayAsync", "hipMemcpyFromArrayAsync"),
    ("cudaMemcpyFromSymbol", "hipMemcpyFromSymbol"),
    ("cudaMemcpyFromSymbolAsync", "hipMemcpyFromSymbolAsync"),
    ("cudaMemAdvise", "hipMemAdvise"),
    ("cudaMemRangeGetAttribute", "hipMemRangeGetAttribute"),
    ("cudaMemRangeGetAttributes", "hipMemRangeGetAttributes"),
    ("cudaMemAdviseSetReadMostly", "hipMemAdviseSetReadMostly"),
    ("cudaMemAdviseUnsetReadMostly", "hipMemAdviseUnsetReadMostly"),
    ("cudaMemAdviseSetPreferredLocation", "hipMemAdviseSetPreferredLocation"),
    ("cudaMemAdviseUnsetPreferredLocation", "hipMemAdviseUnsetPreferredLocation"),
    ("cudaMemAdviseSetAccessedBy", "hipMemAdviseSetAccessedBy"),
    ("cudaMemAdviseUnsetAccessedBy", "hipMemAdviseUnsetAccessedBy"),
    ("cudaMemRangeAttributeReadMostly", "hipMemRangeAttributeReadMostly"),
    ("cudaMemRangeAttributePreferredLocation", "hipMemRangeAttributePreferredLocation"),
    ("cudaMemRangeAttributeAccessedBy", "hipMemRangeAttributeAccessedBy"),
    ("cudaMemRangeAttributeLastPrefetchLocation", "hipMemRangeAttributeLastPrefetchLocation"),
    ("cudaMemcpyHostToHost", "hipMemcpyHostToHost"),
    ("cudaMemcpyHostToDevice", "hipMemcpyHostToDevice"),
    ("cudaMemcpyDeviceToHost", "hipMemcpyDeviceToHost"),
    ("cudaMemcpyDeviceToDevice", "hipMemcpyDeviceToDevice"),
    ("cudaMemcpyDefault", "hipMemcpyDefault"),
    ("cudaMemset", "hipMemset"),
    ("cudaMemsetAsync", "hipMemsetAsync"),
    ("cudaMemset2D", "hipMemset2D"),
    ("cudaMemset2DAsync", "hipMemset2DAsync"),
    ("cudaMemset3D", "hipMemset3D"),
    ("cudaMemset3DAsync", "hipMemset3DAsync"),
    ("cudaMemGetInfo", "hipMemGetInfo"),
    ("cudaDeviceGetDefaultMemPool", "hipDeviceGetDefaultMemPool"),
    ("cudaMemAccessDesc", "hipMemAccessDesc"),
    ("cudaMemAccessFlagsProtReadWrite", "hipMemAccessFlagsProtReadWrite"),
    ("cudaMemLocationTypeDevice", "hipMemLocationTypeDevice"),
    ("cudaMemPoolAttrReleaseThreshold", "hipMemPoolAttrReleaseThreshold"),
    ("cudaMemPoolAttrReservedMemCurrent", "hipMemPoolAttrReservedMemCurrent"),
    ("cudaMemPoolAttrReservedMemHigh", "hipMemPoolAttrReservedMemHigh"),
    ("cudaMemPoolAttrUsedMemCurrent", "hipMemPoolAttrUsedMemCurrent"),
    ("cudaMemPoolAttrUsedMemHigh", "hipMemPoolAttrUsedMemHigh"),
    ("cudaMemPoolGetAttribute", "hipMemPoolGetAttribute"),
    ("cudaMemPoolReuseAllowInternalDependencies", "hipMemPoolReuseAllowInternalDependencies"),
    ("cudaMemPoolReuseAllowOpportunistic", "hipMemPoolReuseAllowOpportunistic"),
    ("cudaMemPoolReuseFollowEventDependencies", "hipMemPoolReuseFollowEventDependencies"),
    ("cudaMemPoolSetAccess", "hipMemPoolSetAccess"),
    ("cudaMemPoolSetAttribute", "hipMemPoolSetAttribute"),
    ("cudaMemPoolTrimTo", "hipMemPoolTrimTo"),
    ("cudaMemPool_t", "hipMemPool_t"),
    ("cudaArrayGetInfo", "hipArrayGetInfo"),
    ("cudaFreeMipmappedArray", "hipFreeMipmappedArray"),
    ("cudaGetMipmappedArrayLevel", "hipGetMipmappedArrayLevel"),
    ("cudaGetSymbolAddress", "hipGetSymbolAddress"),
    ("cudaGetSymbolSize", "hipGetSymbolSize"),
    ("cudaMemPrefetchAsync", "hipMemPrefetchAsync"),
    ("cudaMallocHost", "hipHostMalloc"),
    ("cudaMallocArray", "hipMallocArray"),
    ("cudaMalloc", "hipMalloc"),
    ("cudaMalloc3D", "hipMalloc3D"),
    ("cudaMalloc3DArray", "hipMalloc3DArray"),
    ("cudaMallocAsync", "hipMallocAsync"),
    ("cudaMallocManaged", "hipMallocManaged"),
    ("cudaMallocMipmappedArray", "hipMallocMipmappedArray"),
    ("cudaMallocPitch", "hipMallocPitch"),
    ("cudaFreeHost", "hipHostFree"),
    ("cudaFreeArray", "hipFreeArray"),
    ("cudaFree", "hipFree"),
    ("cudaFreeAsync", "hipFreeAsync"),
    ("cudaHostRegister", "hipHostRegister"),
    ("cudaHostUnregister", "hipHostUnregister"),
    ("cudaHostAlloc", "hipHostMalloc"),
    ("cudaMemoryTypeHost", "hipMemoryTypeHost"),
    ("cudaMemoryTypeDevice", "hipMemoryTypeDevice"),
    ("make_cudaExtent", "make_hipExtent"),
    ("make_cudaPitchedPtr", "make_hipPitchedPtr"),
    ("make_cudaPos", "make_hipPos"),
    ("cudaHostAllocDefault", "hipHostMallocDefault"),
    ("cudaHostAllocPortable", "hipHostMallocPortable"),
    ("cudaHostAllocMapped", "hipHostMallocMapped"),
    ("cudaHostAllocWriteCombined", "hipHostMallocWriteCombined"),
    ("cudaHostFn_t", "hipHostFn_t"),
    ("cudaHostGetFlags", "hipHostGetFlags"),
    ("cudaHostRegisterDefault", "hipHostRegisterDefault"),
    ("cudaHostRegisterPortable", "hipHostRegisterPortable"),
    ("cudaHostRegisterMapped", "hipHostRegisterMapped"),
    ("cudaHostRegisterIoMemory", "hipHostRegisterIoMemory"),
    ("cudaEventCreate", "hipEventCreate"),
    ("cudaEventCreateWithFlags", "hipEventCreateWithFlags"),
    ("cudaEventDestroy", "hipEventDestroy"),
    ("cudaEventRecord", "hipEventRecord"),
    ("cudaEventElapsedTime", "hipEventElapsedTime"),
    ("cudaEventSynchronize", "hipEventSynchronize"),
    ("cudaEventQuery", "hipEventQuery"),
    ("cudaEventDefault", "hipEventDefault"),
    ("cudaEventBlockingSync", "hipEventBlockingSync"),
    ("cudaEventDisableTiming", "hipEventDisableTiming"),
    ("cudaEventInterprocess", "hipEventInterprocess"),
    ("cudaStreamCreate", "hipStreamCreate"),
    ("cudaStreamCreateWithFlags", "hipStreamCreateWithFlags"),
    ("cudaStreamCreateWithPriority", "hipStreamCreateWithPriority"),
    ("cudaStreamDestroy", "hipStreamDestroy"),
    ("cudaStreamWaitEvent", "hipStreamWaitEvent"),
    ("cudaStreamSynchronize", "hipStreamSynchronize"),
    ("cudaStreamGetFlags", "hipStreamGetFlags"),
    ("cudaStreamQuery", "hipStreamQuery"),
    ("cudaStreamAddCallback", "hipStreamAddCallback"),
    ("cudaStreamAttachMemAsync", "hipStreamAttachMemAsync"),
    ("cudaStreamGetPriority", "hipStreamGetPriority"),
    ("cudaCpuDeviceId", "hipCpuDeviceId"),
    ("cudaStreamDefault", "hipStreamDefault"),
    ("cudaStreamNonBlocking", "hipStreamNonBlocking"),
    ("cudaStreamGetCaptureInfo", "hipStreamGetCaptureInfo"),
    ("cudaStreamGetCaptureInfo_v2", "hipStreamGetCaptureInfo_v2"),
    ("cudaStreamCaptureStatus", "hipStreamCaptureStatus"),
    ("cudaStreamCaptureStatusActive", "hipStreamCaptureStatusActive"),
    ("cudaStreamCaptureStatusNone", "hipStreamCaptureStatusNone"),
    ("cudaStreamCaptureStatusInvalidated", "hipStreamCaptureStatusInvalidated"),
    ("cudaStreamCaptureMode", "hipStreamCaptureMode"),
    ("cudaStreamCaptureModeGlobal", "hipStreamCaptureModeGlobal"),
    ("cudaStreamCaptureModeRelaxed", "hipStreamCaptureModeRelaxed"),
    ("cudaStreamCaptureModeThreadLocal", "hipStreamCaptureModeThreadLocal"),
    ("cudaStreamBeginCapture", "hipStreamBeginCapture"),
    ("cudaStreamEndCapture", "hipStreamEndCapture"),
    ("cudaGraphNode_t", "hipGraphNode_t"),
    ("cudaGraphInstantiate", "hipGraphInstantiate"),
    ("cudaGraphInstantiateWithFlags", "hipGraphInstantiateWithFlags"),
    ("cudaGraphInstantiateFlagAutoFreeOnLaunch", "hipGraphInstantiateFlagAutoFreeOnLaunch"),
    ("cudaGraphDestroy", "hipGraphDestroy"),
    ("cudaGraphExecDestroy", "hipGraphExecDestroy"),
    ("cudaGraphLaunch", "hipGraphLaunch"),
    ("cudaGraphGetNodes", "hipGraphGetNodes"),
    ("cudaGraphDebugDotPrint", "hipGraphDebugDotPrint"),
    ("cudaGraphDebugDotFlagsVerbose", "hipGraphDebugDotFlagsVerbose"),
    ("cudaGraphRetainUserObject", "hipGraphRetainUserObject"),
    ("cudaGraphUserObjectMove", "hipGraphUserObjectMove"),
    ("cudaUserObject_t", "hipUserObject_t"),
    ("cudaUserObjectCreate", "hipUserObjectCreate"),
    ("cudaUserObjectNoDestructorSync", "hipUserObjectNoDestructorSync"),
    ("cudaThreadExchangeStreamCaptureMode", "hipThreadExchangeStreamCaptureMode"),
    ("cudaStreamIsCapturing", "hipStreamIsCapturing"),
    ("cudaDeviceSynchronize", "hipDeviceSynchronize"),
    ("cudaDeviceReset", "hipDeviceReset"),
    ("cudaSetDevice", "hipSetDevice"),
    ("cudaGetDevice", "hipGetDevice"),
    ("cudaGetDeviceCount", "hipGetDeviceCount"),
    ("cudaChooseDevice", "hipChooseDevice"),
    ("cudaThreadExit", "hipDeviceReset"),
    ("cudaThreadGetCacheConfig", "hipDeviceGetCacheConfig"),
    ("cudaThreadGetLimit", "hipThreadGetLimit"),
    ("cudaThreadSetCacheConfig", "hipDeviceSetCacheConfig"),
    ("cudaThreadSetLimit", "hipThreadSetLimit"),
    ("cudaThreadSynchronize", "hipDeviceSynchronize"),
    ("cudaDeviceGetAttribute", "hipDeviceGetAttribute"),
    ("cudaDevAttrMaxThreadsPerBlock", "hipDeviceAttributeMaxThreadsPerBlock"),
    ("cudaDevAttrMaxBlocksPerMultiprocessor", "hipDeviceAttributeMaxBlocksPerMultiprocessor"),
    ("cudaDevAttrMaxBlockDimX", "hipDeviceAttributeMaxBlockDimX"),
    ("cudaDevAttrMaxBlockDimY", "hipDeviceAttributeMaxBlockDimY"),
    ("cudaDevAttrMaxBlockDimZ", "hipDeviceAttributeMaxBlockDimZ"),
    ("cudaDevAttrMaxGridDimX", "hipDeviceAttributeMaxGridDimX"),
    ("cudaDevAttrMaxGridDimY", "hipDeviceAttributeMaxGridDimY"),
    ("cudaDevAttrMaxGridDimZ", "hipDeviceAttributeMaxGridDimZ"),
    ("cudaDevAttrMaxSharedMemoryPerBlock", "hipDeviceAttributeMaxSharedMemoryPerBlock"),
    ("cudaDevAttrMaxSharedMemoryPerBlockOptin", "hipDeviceAttributeMaxSharedMemoryPerBlock"),
    ("cudaDevAttrTotalConstantMemory", "hipDeviceAttributeTotalConstantMemory"),
    ("cudaDevAttrWarpSize", "hipDeviceAttributeWarpSize"),
    ("cudaDevAttrMaxPitch", "hipDeviceAttributeMaxPitch"),
    ("cudaDevAttrMaxRegistersPerBlock", "hipDeviceAttributeMaxRegistersPerBlock"),
    ("cudaDevAttrClockRate", "hipDeviceAttributeClockRate"),
    ("cudaDevAttrTextureAlignment", "hipDeviceAttributeTextureAlignment"),
    ("cudaDevAttrGpuOverlap", "hipDeviceAttributeGpuOverlap"),
    ("cudaDevAttrMultiProcessorCount", "hipDeviceAttributeMultiprocessorCount"),
    ("cudaDevAttrKernelExecTimeout", "hipDeviceAttributeKernelExecTimeout"),
    ("cudaDevAttrIntegrated", "hipDeviceAttributeIntegrated"),
    ("cudaDevAttrReserved94", "hipDeviceAttributeCanUseStreamWaitValue"),
    ("cudaDevAttrCooperativeLaunch", "hipDeviceAttributeCooperativeLaunch"),
    ("cudaDevAttrCooperativeMultiDeviceLaunch", "hipDeviceAttributeCooperativeMultiDeviceLaunch"),
    ("cudaDevAttrCanMapHostMemory", "hipDeviceAttributeCanMapHostMemory"),
    ("cudaDevAttrComputeMode", "hipDeviceAttributeComputeMode"),
    ("cudaDevAttrMaxTexture1DWidth", "hipDeviceAttributeMaxTexture1DWidth"),
    ("cudaDevAttrMaxTexture2DWidth", "hipDeviceAttributeMaxTexture2DWidth"),
    ("cudaDevAttrMaxTexture2DHeight", "hipDeviceAttributeMaxTexture2DHeight"),
    ("cudaDevAttrMaxTexture3DWidth", "hipDeviceAttributeMaxTexture3DWidth"),
    ("cudaDevAttrMaxTexture3DHeight", "hipDeviceAttributeMaxTexture3DHeight"),
    ("cudaDevAttrMaxTexture3DDepth", "hipDeviceAttributeMaxTexture3DDepth"),
    ("cudaDevAttrMaxTexture2DLayeredWidth", "hipDeviceAttributeMaxTexture2DLayeredWidth"),
    ("cudaDevAttrMaxTexture2DLayeredHeight", "hipDeviceAttributeMaxTexture2DLayeredHeight"),
    ("cudaDevAttrMaxTexture2DLayeredLayers", "hipDeviceAttributeMaxTexture2DLayeredLayers"),
    ("cudaDevAttrSurfaceAlignment", "hipDeviceAttributeSurfaceAlignment"),
    ("cudaDevAttrConcurrentKernels", "hipDeviceAttributeConcurrentKernels"),
    ("cudaDevAttrEccEnabled", "hipDeviceAttributeEccEnabled"),
    ("cudaDevAttrMemoryPoolsSupported", "hipDeviceAttributeMemoryPoolsSupported"),
    ("cudaDevAttrPciBusId", "hipDeviceAttributePciBusId"),
    ("cudaDevAttrPciDeviceId", "hipDeviceAttributePciDeviceId"),
    ("cudaDevAttrTccDriver", "hipDeviceAttributeTccDriver"),
    ("cudaDevAttrMemoryClockRate", "hipDeviceAttributeMemoryClockRate"),
    ("cudaDevAttrGlobalMemoryBusWidth", "hipDeviceAttributeMemoryBusWidth"),
    ("cudaDevAttrL2CacheSize", "hipDeviceAttributeL2CacheSize"),
    ("cudaDevAttrMaxThreadsPerMultiProcessor", "hipDeviceAttributeMaxThreadsPerMultiProcessor"),
    ("cudaDevAttrAsyncEngineCount", "hipDeviceAttributeAsyncEngineCount"),
    ("cudaDevAttrUnifiedAddressing", "hipDeviceAttributeUnifiedAddressing"),
    ("cudaDevAttrMaxTexture1DLayeredWidth", "hipDeviceAttributeMaxTexture1DLayeredWidth"),
    ("cudaDevAttrMaxTexture1DLayeredLayers", "hipDeviceAttributeMaxTexture1DLayeredLayers"),
    ("cudaDevAttrMaxTexture2DGatherWidth", "hipDeviceAttributeMaxTexture2DGatherWidth"),
    ("cudaDevAttrMaxTexture2DGatherHeight", "hipDeviceAttributeMaxTexture2DGatherHeight"),
    ("cudaDevAttrMaxTexture3DWidthAlt", "hipDeviceAttributeMaxTexture3DWidthAlternate"),
    ("cudaDevAttrMaxTexture3DHeightAlt", "hipDeviceAttributeMaxTexture3DHeightAlternate"),
    ("cudaDevAttrMaxTexture3DDepthAlt", "hipDeviceAttributeMaxTexture3DDepthAlternate"),
    ("cudaDevAttrPciDomainId", "hipDeviceAttributePciDomainId"),
    ("cudaDevAttrTexturePitchAlignment", "hipDeviceAttributeTexturePitchAlignment"),
    ("cudaDevAttrMaxTextureCubemapWidth", "hipDeviceAttributeMaxTextureCubemapWidth"),
    ("cudaDevAttrMaxTextureCubemapLayeredWidth", "hipDeviceAttributeMaxTextureCubemapLayeredWidth"),
    ("cudaDevAttrMaxTextureCubemapLayeredLayers", "hipDeviceAttributeMaxTextureCubemapLayeredLayers"),
    ("cudaDevAttrMaxSurface1DWidth", "hipDeviceAttributeMaxSurface1DWidth"),
    ("cudaDevAttrMaxSurface2DWidth", "hipDeviceAttributeMaxSurface2DWidth"),
    ("cudaDevAttrMaxSurface2DHeight", "hipDeviceAttributeMaxSurface2DHeight"),
    ("cudaDevAttrMaxSurface3DWidth", "hipDeviceAttributeMaxSurface3DWidth"),
    ("cudaDevAttrMaxSurface3DHeight", "hipDeviceAttributeMaxSurface3DHeight"),
    ("cudaDevAttrMaxSurface3DDepth", "hipDeviceAttributeMaxSurface3DDepth"),
    ("cudaDevAttrMaxSurface1DLayeredWidth", "hipDeviceAttributeMaxSurface1DLayeredWidth"),
    ("cudaDevAttrMaxSurface1DLayeredLayers", "hipDeviceAttributeMaxSurface1DLayeredLayers"),
    ("cudaDevAttrMaxSurface2DLayeredWidth", "hipDeviceAttributeMaxSurface2DLayeredWidth"),
    ("cudaDevAttrMaxSurface2DLayeredHeight", "hipDeviceAttributeMaxSurface2DLayeredHeight"),
    ("cudaDevAttrMaxSurface2DLayeredLayers", "hipDeviceAttributeMaxSurface2DLayeredLayers"),
    ("cudaDevAttrMaxSurfaceCubemapWidth", "hipDeviceAttributeMaxSurfaceCubemapWidth"),
    ("cudaDevAttrMaxSurfaceCubemapLayeredWidth", "hipDeviceAttributeMaxSurfaceCubemapLayeredWidth"),
    ("cudaDevAttrMaxSurfaceCubemapLayeredLayers", "hipDeviceAttributeMaxSurfaceCubemapLayeredLayers"),
    ("cudaDevAttrMaxTexture1DLinearWidth", "hipDeviceAttributeMaxTexture1DLinearWidth"),
    ("cudaDevAttrMaxTexture2DLinearWidth", "hipDeviceAttributeMaxTexture2DLinearWidth"),
    ("cudaDevAttrMaxTexture2DLinearHeight", "hipDeviceAttributeMaxTexture2DLinearHeight"),
    ("cudaDevAttrMaxTexture2DLinearPitch", "hipDeviceAttributeMaxTexture2DLinearPitch"),
    ("cudaDevAttrMaxTexture2DMipmappedWidth", "hipDeviceAttributeMaxTexture2DMipmappedWidth"),
    ("cudaDevAttrMaxTexture2DMipmappedHeight", "hipDeviceAttributeMaxTexture2DMipmappedHeight"),
    ("cudaDevAttrComputeCapabilityMajor", "hipDeviceAttributeComputeCapabilityMajor"),
    ("cudaDevAttrComputeCapabilityMinor", "hipDeviceAttributeComputeCapabilityMinor"),
    ("cudaDevAttrMaxTexture1DMipmappedWidth", "hipDeviceAttributeMaxTexture1DMipmappedWidth"),
    ("cudaDevAttrStreamPrioritiesSupported", "hipDeviceAttributeStreamPrioritiesSupported"),
    ("cudaDevAttrGlobalL1CacheSupported", "hipDeviceAttributeGlobalL1CacheSupported"),
    ("cudaDevAttrLocalL1CacheSupported", "hipDeviceAttributeLocalL1CacheSupported"),
    ("cudaDevAttrMaxSharedMemoryPerMultiprocessor", "hipDeviceAttributeMaxSharedMemoryPerMultiprocessor"),
    ("cudaDevAttrHostRegisterSupported", "hipDeviceAttributeHostRegisterSupported"),
    ("cudaDevAttrMaxRegistersPerMultiprocessor", "hipDeviceAttributeMaxRegistersPerMultiprocessor"),
    ("cudaDevAttrManagedMemory", "hipDeviceAttributeManagedMemory"),
    ("cudaDevAttrDirectManagedMemAccessFromHost", "hipDeviceAttributeDirectManagedMemAccessFromHost"),
    ("cudaDevAttrIsMultiGpuBoard", "hipDeviceAttributeIsMultiGpuBoard"),
    ("cudaDevAttrMultiGpuBoardGroupID", "hipDeviceAttributeMultiGpuBoardGroupID"),
    ("cudaDevAttrHostNativeAtomicSupported", "hipDeviceAttributeHostNativeAtomicSupported"),
    ("cudaDevAttrSingleToDoublePrecisionPerfRatio", "hipDeviceAttributeSingleToDoublePrecisionPerfRatio"),
    ("cudaDevAttrPageableMemoryAccess", "hipDeviceAttributePageableMemoryAccess"),
    ("cudaDevAttrPageableMemoryAccessUsesHostPageTables", "hipDeviceAttributePageableMemoryAccessUsesHostPageTables"),
    ("cudaDevAttrConcurrentManagedAccess", "hipDeviceAttributeConcurrentManagedAccess"),
    ("cudaDevAttrComputePreemptionSupported", "hipDeviceAttributeComputePreemptionSupported"),
    ("cudaDevAttrCanUseHostPointerForRegisteredMem", "hipDeviceAttributeCanUseHostPointerForRegisteredMem"),
    ("cudaPointerGetAttributes", "hipPointerGetAttributes"),
    ("cudaHostGetDevicePointer", "hipHostGetDevicePointer"),
    ("cudaGetDeviceProperties", "hipGetDeviceProperties"),
    ("cudaDeviceGetPCIBusId", "hipDeviceGetPCIBusId"),
    ("cudaDeviceGetByPCIBusId", "hipDeviceGetByPCIBusId"),
    ("cudaDeviceGetStreamPriorityRange", "hipDeviceGetStreamPriorityRange"),
    ("cudaSetValidDevices", "hipSetValidDevices"),
    ("cudaDevP2PAttrPerformanceRank", "hipDeviceP2PAttributePerformanceRank"),
    ("cudaDevP2PAttrAccessSupported", "hipDeviceP2PAttributeAccessSupported"),
    ("cudaDevP2PAttrNativeAtomicSupported", "hipDeviceP2PAttributeNativeAtomicSupported"),
    ("cudaDeviceGetP2PAttribute", "hipDeviceGetP2PAttribute"),
    ("cudaComputeModeDefault", "hipComputeModeDefault"),
    ("cudaComputeModeExclusive", "hipComputeModeExclusive"),
    ("cudaComputeModeProhibited", "hipComputeModeProhibited"),
    ("cudaComputeModeExclusiveProcess", "hipComputeModeExclusiveProcess"),
    ("cudaGetDeviceFlags", "hipGetDeviceFlags"),
    ("cudaSetDeviceFlags", "hipSetDeviceFlags"),
    ("cudaDeviceScheduleAuto", "hipDeviceScheduleAuto"),
    ("cudaDeviceScheduleSpin", "hipDeviceScheduleSpin"),
    ("cudaDeviceScheduleYield", "hipDeviceScheduleYield"),
    ("cudaDeviceBlockingSync", "hipDeviceScheduleBlockingSync"),
    ("cudaDeviceScheduleBlockingSync", "hipDeviceScheduleBlockingSync"),
    ("cudaDeviceScheduleMask", "hipDeviceScheduleMask"),
    ("cudaDeviceMapHost", "hipDeviceMapHost"),
    ("cudaDeviceLmemResizeToMax", "hipDeviceLmemResizeToMax"),
    ("cudaDeviceMask", "hipDeviceMask"),
    ("cudaDeviceSetCacheConfig", "hipDeviceSetCacheConfig"),
    ("cudaDeviceGetCacheConfig", "hipDeviceGetCacheConfig"),
    ("cudaFuncAttributes", "hipFuncAttributes"),
    ("cudaFuncAttributeMaxDynamicSharedMemorySize", "hipFuncAttributeMaxDynamicSharedMemorySize"),
    ("cudaFuncAttributePreferredSharedMemoryCarveout", "hipFuncAttributePreferredSharedMemoryCarveout"),
    ("cudaFuncSetAttribute", "hipFuncSetAttribute"),
    ("cudaFuncSetCacheConfig", "hipFuncSetCacheConfig"),
    ("cudaFuncCachePreferNone", "hipFuncCachePreferNone"),
    ("cudaFuncCachePreferShared", "hipFuncCachePreferShared"),
    ("cudaFuncCachePreferL1", "hipFuncCachePreferL1"),
    ("cudaFuncCachePreferEqual", "hipFuncCachePreferEqual"),
    ("cudaFuncGetAttributes", "hipFuncGetAttributes"),
    ("cudaFuncSetSharedMemConfig", "hipFuncSetSharedMemConfig"),
    ("cudaGetParameterBuffer", "hipGetParameterBuffer"),
    ("cudaSetDoubleForDevice", "hipSetDoubleForDevice"),
    ("cudaSetDoubleForHost", "hipSetDoubleForHost"),
    ("cudaConfigureCall", "hipConfigureCall"),
    ("cudaLaunch", "hipLaunch"),
    ("cudaLaunchCooperativeKernel", "hipLaunchCooperativeKernel"),
    ("cudaLaunchHostFunc", "hipLaunchHostFunc"),
    ("cudaSetupArgument", "hipSetupArgument"),
    ("cudaDriverGetVersion", "hipDriverGetVersion"),
    ("cudaRuntimeGetVersion", "hipRuntimeGetVersion"),
    ("cudaOccupancyMaxPotentialBlockSize", "hipOccupancyMaxPotentialBlockSize"),
    ("cudaErrorContextIsDestroyed", "hipErrorContextIsDestroyed"),
    ("cudaDeviceSetLimit", "hipDeviceSetLimit"),
    ("cudaMallocFromPoolAsync", "hipMallocFromPoolAsync"),
    ("cudaDeviceGetMemPool", "hipDeviceGetMemPool"),
    ("cudaDeviceSetMemPool", "hipDeviceSetMemPool"),
    ("cudaMemPoolAttr", "hipMemPoolAttr"),
    ("cudaMemPoolProps", "hipMemPoolProps"),
    ("cudaMemAllocationTypePinned", "hipMemAllocationTypePinned"),
    ("cudaMemHandleTypeNone", "hipMemHandleTypeNone"),
    ("cudaMemHandleTypePosixFileDescriptor", "hipMemHandleTypePosixFileDescriptor"),
    ("cudaMemLocation", "hipMemLocation"),
    ("cudaMemPoolCreate", "hipMemPoolCreate"),
    ("cudaMemPoolDestroy", "hipMemPoolDestroy"),
    ("cudaCreateSurfaceObject", "hipCreateSurfaceObject"),
    ("cudaDestroySurfaceObject", "hipDestroySurfaceObject"),
    ("cudaGraphUpload", "hipGraphUpload"),
    ("cudaOccupancyMaxPotentialBlockSizeWithFlags", "hipOccupancyMaxPotentialBlockSizeWithFlags"),
    ("cudaOccupancyMaxActiveBlocksPerMultiprocessor", "hipOccupancyMaxActiveBlocksPerMultiprocessor"),
    ("cudaOccupancyMaxActiveBlocksPerMultiprocessorWithFlags", "hipOccupancyMaxActiveBlocksPerMultiprocessorWithFlags"),
    ("cudaOccupancyMaxPotentialBlockSizeVariableSMem", "hipOccupancyMaxPotentialBlockSizeVariableSMem"),
    ("cudaOccupancyMaxPotentialBlockSizeVariableSMemWithFlags", "hipOccupancyMaxPotentialBlockSizeVariableSMemWithFlags"),
    ("cudaDeviceCanAccessPeer", "hipDeviceCanAccessPeer"),
    ("cudaDeviceDisablePeerAccess", "hipDeviceDisablePeerAccess"),
    ("cudaDeviceEnablePeerAccess", "hipDeviceEnablePeerAccess"),
    ("cudaMemcpyPeerAsync", "hipMemcpyPeerAsync"),
    ("cudaMemcpyPeer", "hipMemcpyPeer"),
    ("cudaIpcMemLazyEnablePeerAccess", "hipIpcMemLazyEnablePeerAccess"),
    ("cudaDeviceSetSharedMemConfig", "hipDeviceSetSharedMemConfig"),
    ("cudaDeviceGetSharedMemConfig", "hipDeviceGetSharedMemConfig"),
    ("cudaSharedMemBankSizeDefault", "hipSharedMemBankSizeDefault"),
    ("cudaSharedMemBankSizeFourByte", "hipSharedMemBankSizeFourByte"),
    ("cudaSharedMemBankSizeEightByte", "hipSharedMemBankSizeEightByte"),
    ("cudaLimitStackSize", "hipLimitStackSize"),
    ("cudaLimitPrintfFifoSize", "hipLimitPrintfFifoSize"),
    ("cudaLimitMallocHeapSize", "hipLimitMallocHeapSize"),
    ("cudaLimitDevRuntimeSyncDepth", "hipLimitDevRuntimeSyncDepth"),
    ("cudaLimitDevRuntimePendingLaunchCount", "hipLimitDevRuntimePendingLaunchCount"),
    ("cudaDeviceGetLimit", "hipDeviceGetLimit"),
    ("cudaProfilerInitialize", "hipProfilerInitialize"),
    ("cudaProfilerStart", "hipProfilerStart"),
    ("cudaProfilerStop", "hipProfilerStop"),
    ("cudaKeyValuePair", "hipKeyValuePair"),
    ("cudaCSV", "hipCSV"),
    ("cudaReadModeElementType", "hipReadModeElementType"),
    ("cudaReadModeNormalizedFloat", "hipReadModeNormalizedFloat"),
    ("cudaFilterModePoint", "hipFilterModePoint"),
    ("cudaFilterModeLinear", "hipFilterModeLinear"),
    ("cudaBindTexture", "hipBindTexture"),
    ("cudaUnbindTexture", "hipUnbindTexture"),
    ("cudaBindTexture2D", "hipBindTexture2D"),
    ("cudaBindTextureToArray", "hipBindTextureToArray"),
    ("cudaBindTextureToMipmappedArray", "hipBindTextureToMipmappedArray"),
    ("cudaGetTextureAlignmentOffset", "hipGetTextureAlignmentOffset"),
    ("cudaGetTextureReference", "hipGetTextureReference"),
    ("cudaChannelFormatKindSigned", "hipChannelFormatKindSigned"),
    ("cudaChannelFormatKindUnsigned", "hipChannelFormatKindUnsigned"),
    ("cudaChannelFormatKindFloat", "hipChannelFormatKindFloat"),
    ("cudaChannelFormatKindNone", "hipChannelFormatKindNone"),
    ("cudaCreateChannelDesc", "hipCreateChannelDesc"),
    ("cudaGetChannelDesc", "hipGetChannelDesc"),
    ("cudaResourceTypeArray", "hipResourceTypeArray"),
    ("cudaResourceTypeMipmappedArray", "hipResourceTypeMipmappedArray"),
    ("cudaResourceTypeLinear", "hipResourceTypeLinear"),
    ("cudaResourceTypePitch2D", "hipResourceTypePitch2D"),
    ("cudaResViewFormatNone", "hipResViewFormatNone"),
    ("cudaResViewFormatUnsignedChar1", "hipResViewFormatUnsignedChar1"),
    ("cudaResViewFormatUnsignedChar2", "hipResViewFormatUnsignedChar2"),
    ("cudaResViewFormatUnsignedChar4", "hipResViewFormatUnsignedChar4"),
    ("cudaResViewFormatSignedChar1", "hipResViewFormatSignedChar1"),
    ("cudaResViewFormatSignedChar2", "hipResViewFormatSignedChar2"),
    ("cudaResViewFormatSignedChar4", "hipResViewFormatSignedChar4"),
    ("cudaResViewFormatUnsignedShort1", "hipResViewFormatUnsignedShort1"),
    ("cudaResViewFormatUnsignedShort2", "hipResViewFormatUnsignedShort2"),
    ("cudaResViewFormatUnsignedShort4", "hipResViewFormatUnsignedShort4"),
    ("cudaResViewFormatSignedShort1", "hipResViewFormatSignedShort1"),
    ("cudaResViewFormatSignedShort2", "hipResViewFormatSignedShort2"),
    ("cudaResViewFormatSignedShort4", "hipResViewFormatSignedShort4"),
    ("cudaResViewFormatUnsignedInt1", "hipResViewFormatUnsignedInt1"),
    ("cudaResViewFormatUnsignedInt2", "hipResViewFormatUnsignedInt2"),
    ("cudaResViewFormatUnsignedInt4", "hipResViewFormatUnsignedInt4"),
    ("cudaResViewFormatSignedInt1", "hipResViewFormatSignedInt1"),
    ("cudaResViewFormatSignedInt2", "hipResViewFormatSignedInt2"),
    ("cudaResViewFormatSignedInt4", "hipResViewFormatSignedInt4"),
    ("cudaResViewFormatHalf1", "hipResViewFormatHalf1"),
    ("cudaResViewFormatHalf2", "hipResViewFormatHalf2"),
    ("cudaResViewFormatHalf4", "hipResViewFormatHalf4"),
    ("cudaResViewFormatFloat1", "hipResViewFormatFloat1"),
    ("cudaResViewFormatFloat2", "hipResViewFormatFloat2"),
    ("cudaResViewFormatFloat4", "hipResViewFormatFloat4"),
    ("cudaResViewFormatUnsignedBlockCompressed1", "hipResViewFormatUnsignedBlockCompressed1"),
    ("cudaResViewFormatUnsignedBlockCompressed2", "hipResViewFormatUnsignedBlockCompressed2"),
    ("cudaResViewFormatUnsignedBlockCompressed3", "hipResViewFormatUnsignedBlockCompressed3"),
    ("cudaResViewFormatUnsignedBlockCompressed4", "hipResViewFormatUnsignedBlockCompressed4"),
    ("cudaResViewFormatSignedBlockCompressed4", "hipResViewFormatSignedBlockCompressed4"),
    ("cudaResViewFormatUnsignedBlockCompressed5", "hipResViewFormatUnsignedBlockCompressed5"),
    ("cudaResViewFormatSignedBlockCompressed5", "hipResViewFormatSignedBlockCompressed5"),
    ("cudaResViewFormatUnsignedBlockCompressed6H", "hipResViewFormatUnsignedBlockCompressed6H"),
    ("cudaResViewFormatSignedBlockCompressed6H", "hipResViewFormatSignedBlockCompressed6H"),
    ("cudaResViewFormatUnsignedBlockCompressed7", "hipResViewFormatUnsignedBlockCompressed7"),
    ("cudaAddressModeWrap", "hipAddressModeWrap"),
    ("cudaAddressModeClamp", "hipAddressModeClamp"),
    ("cudaAddressModeMirror", "hipAddressModeMirror"),
    ("cudaAddressModeBorder", "hipAddressModeBorder"),
    ("cudaCreateTextureObject", "hipCreateTextureObject"),
    ("cudaDestroyTextureObject", "hipDestroyTextureObject"),
    ("cudaGetTextureObjectResourceDesc", "hipGetTextureObjectResourceDesc"),
    ("cudaGetTextureObjectResourceViewDesc", "hipGetTextureObjectResourceViewDesc"),
    ("cudaGetTextureObjectTextureDesc", "hipGetTextureObjectTextureDesc"),
    ("cudaBindSurfaceToArray", "hipBindSurfaceToArray"),
    ("cudaGetSurfaceReference", "hipGetSurfaceReference"),
    ("cudaBoundaryModeZero", "hipBoundaryModeZero"),
    ("cudaBoundaryModeClamp", "hipBoundaryModeClamp"),
    ("cudaBoundaryModeTrap", "hipBoundaryModeTrap"),
    ("cudaFormatModeForced", "hipFormatModeForced"),
    ("cudaFormatModeAuto", "hipFormatModeAuto"),
    ("cudaGetSurfaceObjectResourceDesc", "hipGetSurfaceObjectResourceDesc"),
    ("cudaIpcCloseMemHandle", "hipIpcCloseMemHandle"),
    ("cudaIpcGetEventHandle", "hipIpcGetEventHandle"),
    ("cudaIpcGetMemHandle", "hipIpcGetMemHandle"),
    ("cudaIpcOpenEventHandle", "hipIpcOpenEventHandle"),
    ("cudaIpcOpenMemHandle", "hipIpcOpenMemHandle"),
    ("cudaGLGetDevices", "hipGLGetDevices"),
    ("cudaGraphicsGLRegisterBuffer", "hipGraphicsGLRegisterBuffer"),
    ("cudaGraphicsGLRegisterImage", "hipGraphicsGLRegisterImage"),
    ("cudaWGLGetDevice", "hipWGLGetDevice"),
    ("cudaGraphicsMapResources", "hipGraphicsMapResources"),
    ("cudaGraphicsResourceGetMappedMipmappedArray", "hipGraphicsResourceGetMappedMipmappedArray"),
    ("cudaGraphicsResourceGetMappedPointer", "hipGraphicsResourceGetMappedPointer"),
    ("cudaGraphicsResourceSetMapFlags", "hipGraphicsResourceSetMapFlags"),
    ("cudaGraphicsSubResourceGetMappedArray", "hipGraphicsSubResourceGetMappedArray"),
    ("cudaGraphicsUnmapResources", "hipGraphicsUnmapResources"),
    ("cudaGraphicsUnregisterResource", "hipGraphicsUnregisterResource"),
    ("cudaGraphicsCubeFacePositiveX", "hipGraphicsCubeFacePositiveX"),
    ("cudaGraphicsCubeFaceNegativeX", "hipGraphicsCubeFaceNegativeX"),
    ("cudaGraphicsCubeFacePositiveY", "hipGraphicsCubeFacePositiveY"),
    ("cudaGraphicsCubeFaceNegativeY", "hipGraphicsCubeFaceNegativeY"),
    ("cudaGraphicsCubeFacePositiveZ", "hipGraphicsCubeFacePositiveZ"),
    ("cudaGraphicsCubeFaceNegativeZ", "hipGraphicsCubeFaceNegativeZ"),
    ("cudaGraphicsMapFlagsNone", "hipGraphicsMapFlagsNone"),
    ("cudaGraphicsMapFlagsReadOnly", "hipGraphicsMapFlagsReadOnly"),
    ("cudaGraphicsMapFlagsWriteDiscard", "hipGraphicsMapFlagsWriteDiscard"),
    ("cudaGraphicsRegisterFlagsNone", "hipGraphicsRegisterFlagsNone"),
    ("cudaGraphicsRegisterFlagsReadOnly", "hipGraphicsRegisterFlagsReadOnly"),
    ("cudaGraphicsRegisterFlagsWriteDiscard", "hipGraphicsRegisterFlagsWriteDiscard"),
    ("cudaGraphicsRegisterFlagsSurfaceLoadStore", "hipGraphicsRegisterFlagsSurfaceLoadStore"),
    ("cudaGraphicsRegisterFlagsTextureGather", "hipGraphicsRegisterFlagsTextureGather"),
    ("cudaGLDeviceListAll", "HIP_GL_DEVICE_LIST_ALL"),
    ("cudaGLDeviceListCurrentFrame", "HIP_GL_DEVICE_LIST_CURRENT_FRAME"),
    ("cudaGLDeviceListNextFrame", "HIP_GL_DEVICE_LIST_NEXT_FRAME"),
    ("cudaGLMapFlagsNone", "HIP_GL_MAP_RESOURCE_FLAGS_NONE"),
    ("cudaGLMapFlagsReadOnly", "HIP_GL_MAP_RESOURCE_FLAGS_READ_ONLY"),
    ("cudaGLMapFlagsWriteDiscard", "HIP_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD"),
    ("cudaGLMapBufferObject", "hipGLMapBufferObject__"),
    ("cudaGLMapBufferObjectAsync", "hipGLMapBufferObjectAsync__"),
    ("cudaGLRegisterBufferObject", "hipGLRegisterBufferObject"),
    ("cudaGLSetBufferObjectMapFlags", "hipGLSetBufferObjectMapFlags"),
    ("cudaGLSetGLDevice", "hipGLSetGLDevice"),
    ("cudaGLUnmapBufferObject", "hipGLUnmapBufferObject"),
    ("cudaGLUnmapBufferObjectAsync", "hipGLUnmapBufferObjectAsync"),
    ("cudaGLUnregisterBufferObject", "hipGLUnregisterBufferObject"),
    ("cudaD3D9DeviceListAll", "HIP_D3D9_DEVICE_LIST_ALL"),
    ("cudaD3D9DeviceListCurrentFrame", "HIP_D3D9_DEVICE_LIST_CURRENT_FRAME"),
    ("cudaD3D9DeviceListNextFrame", "HIP_D3D9_DEVICE_LIST_NEXT_FRAME"),
    ("cudaD3D9GetDevice", "hipD3D9GetDevice"),
    ("cudaD3D9GetDevices", "hipD3D9GetDevices"),
    ("cudaD3D9GetDirect3DDevice", "hipD3D9GetDirect3DDevice"),
    ("cudaD3D9SetDirect3DDevice", "hipD3D9SetDirect3DDevice"),
    ("cudaGraphicsD3D9RegisterResource", "hipGraphicsD3D9RegisterResource"),
    ("cudaD3D9MapFlags", "hipD3D9MapFlags"),
    ("cudaD3D9MapFlagsNone", "HIP_D3D9_MAPRESOURCE_FLAGS_NONE"),
    ("cudaD3D9MapFlagsReadOnly", "HIP_D3D9_MAPRESOURCE_FLAGS_READONLY"),
    ("cudaD3D9MapFlagsWriteDiscard", "HIP_D3D9_MAPRESOURCE_FLAGS_WRITEDISCARD"),
    ("cudaD3D9RegisterFlagsNone", "HIP_D3D9_REGISTER_FLAGS_NONE"),
    ("cudaD3D9RegisterFlagsArray", "HIP_D3D9_REGISTER_FLAGS_ARRAY"),
    ("cudaD3D9MapResources", "hipD3D9MapResources"),
    ("cudaD3D9RegisterResource", "hipD3D9RegisterResource"),
    ("cudaD3D9ResourceGetMappedArray", "hipD3D9ResourceGetMappedArray"),
    ("cudaD3D9ResourceGetMappedPitch", "hipD3D9ResourceGetMappedPitch"),
    ("cudaD3D9ResourceGetMappedPointer", "hipD3D9ResourceGetMappedPointer"),
    ("cudaD3D9ResourceGetMappedSize", "hipD3D9ResourceGetMappedSize"),
    ("cudaD3D9ResourceGetSurfaceDimensions", "hipD3D9ResourceGetSurfaceDimensions"),
    ("cudaD3D9ResourceSetMapFlags", "hipD3D9ResourceSetMapFlags"),
    ("cudaD3D9UnmapResources", "hipD3D9UnmapResources"),
    ("cudaD3D9UnregisterResource", "hipD3D9UnregisterResource"),
    ("cudaD3D10DeviceListAll", "HIP_D3D10_DEVICE_LIST_ALL"),
    ("cudaD3D10DeviceListCurrentFrame", "HIP_D3D10_DEVICE_LIST_CURRENT_FRAME"),
    ("cudaD3D10DeviceListNextFrame", "HIP_D3D10_DEVICE_LIST_NEXT_FRAME"),
    ("cudaD3D10GetDevice", "hipD3D10GetDevice"),
    ("cudaD3D10GetDevices", "hipD3D10GetDevices"),
    ("cudaGraphicsD3D10RegisterResource", "hipGraphicsD3D10RegisterResource"),
    ("cudaD3D10MapFlagsNone", "HIP_D3D10_MAPRESOURCE_FLAGS_NONE"),
    ("cudaD3D10MapFlagsReadOnly", "HIP_D3D10_MAPRESOURCE_FLAGS_READONLY"),
    ("cudaD3D10MapFlagsWriteDiscard", "HIP_D3D10_MAPRESOURCE_FLAGS_WRITEDISCARD"),
    ("cudaD3D10RegisterFlagsNone", "HIP_D3D10_REGISTER_FLAGS_NONE"),
    ("cudaD3D10RegisterFlagsArray", "HIP_D3D10_REGISTER_FLAGS_ARRAY"),
    ("cudaD3D10GetDirect3DDevice", "hipD3D10GetDirect3DDevice"),
    ("cudaD3D10MapResources", "hipD3D10MapResources"),
    ("cudaD3D10RegisterResource", "hipD3D10RegisterResource"),
    ("cudaD3D10ResourceGetMappedArray", "hipD3D10ResourceGetMappedArray"),
    ("cudaD3D10ResourceGetMappedPitch", "hipD3D10ResourceGetMappedPitch"),
    ("cudaD3D10ResourceGetMappedPointer", "hipD3D10ResourceGetMappedPointer"),
    ("cudaD3D10ResourceGetMappedSize", "hipD3D10ResourceGetMappedSize"),
    ("cudaD3D10ResourceGetSurfaceDimensions", "hipD3D10ResourceGetSurfaceDimensions"),
    ("cudaD3D10ResourceSetMapFlags", "hipD3D10ResourceSetMapFlags"),
    ("cudaD3D10SetDirect3DDevice", "hipD3D10SetDirect3DDevice"),
    ("cudaD3D10UnmapResources", "hipD3D10UnmapResources"),
    ("cudaD3D10UnregisterResource", "hipD3D10UnregisterResource"),
    ("cudaD3D11DeviceListAll", "HIP_D3D11_DEVICE_LIST_ALL"),
    ("cudaD3D11DeviceListCurrentFrame", "HIP_D3D11_DEVICE_LIST_CURRENT_FRAME"),
    ("cudaD3D11DeviceListNextFrame", "HIP_D3D11_DEVICE_LIST_NEXT_FRAME"),
    ("cudaD3D11GetDevice", "hipD3D11GetDevice"),
    ("cudaD3D11GetDevices", "hipD3D11GetDevices"),
    ("cudaGraphicsD3D11RegisterResource", "hipGraphicsD3D11RegisterResource"),
    ("cudaGraphicsVDPAURegisterOutputSurface", "hipGraphicsVDPAURegisterOutputSurface"),
    ("cudaGraphicsVDPAURegisterVideoSurface", "hipGraphicsVDPAURegisterVideoSurface"),
    ("cudaVDPAUGetDevice", "hipVDPAUGetDevice"),
    ("cudaVDPAUSetVDPAUDevice", "hipVDPAUSetDevice"),
    ("cudaEGLStreamConsumerAcquireFrame", "hipEGLStreamConsumerAcquireFrame"),
    ("cudaEGLStreamConsumerConnect", "hipEGLStreamConsumerConnect"),
    ("cudaEGLStreamConsumerConnectWithFlags", "hipEGLStreamConsumerConnectWithFlags"),
    ("cudaEGLStreamConsumerReleaseFrame", "hipEGLStreamConsumerReleaseFrame"),
    ("cudaEGLStreamProducerConnect", "hipEGLStreamProducerConnect"),
    ("cudaEGLStreamProducerDisconnect", "hipEGLStreamProducerDisconnect"),
    ("cudaEGLStreamProducerPresentFrame", "hipEGLStreamProducerPresentFrame"),
    ("cudaEGLStreamProducerReturnFrame", "hipEGLStreamProducerReturnFrame"),
    ("cudaGraphicsEGLRegisterImage", "hipGraphicsEGLRegisterImage"),
    ("cudaGraphicsResourceGetMappedEglFrame", "hipGraphicsResourceGetMappedEglFrame"),
    ("cublasInit", "hipblasInit"),
    ("cublasShutdown", "hipblasShutdown"),
    ("cublasGetVersion", "hipblasGetVersion"),
    ("cublasGetError", "hipblasGetError"),
    ("cublasAlloc", "hipblasAlloc"),
    ("cublasFree", "hipblasFree"),
    ("cublasSetKernelStream", "hipblasSetKernelStream"),
    ("cublasGetAtomicsMode", "hipblasGetAtomicsMode"),
    ("cublasSetAtomicsMode", "hipblasSetAtomicsMode"),
    ("cublasGetMathMode", "hipblasGetMathMode"),
    ("cublasSetMathMode", "hipblasSetMathMode"),
    ("CUBLAS_OP_N", "HIPBLAS_OP_N"),
    ("CUBLAS_OP_T", "HIPBLAS_OP_T"),
    ("CUBLAS_OP_C", "HIPBLAS_OP_C"),
    ("CUBLAS_STATUS_SUCCESS", "HIPBLAS_STATUS_SUCCESS"),
    ("CUBLAS_STATUS_NOT_INITIALIZED", "HIPBLAS_STATUS_NOT_INITIALIZED"),
    ("CUBLAS_STATUS_ALLOC_FAILED", "HIPBLAS_STATUS_ALLOC_FAILED"),
    ("CUBLAS_STATUS_INVALID_VALUE", "HIPBLAS_STATUS_INVALID_VALUE"),
    ("CUBLAS_STATUS_MAPPING_ERROR", "HIPBLAS_STATUS_MAPPING_ERROR"),
    ("CUBLAS_STATUS_EXECUTION_FAILED", "HIPBLAS_STATUS_EXECUTION_FAILED"),
    ("CUBLAS_STATUS_INTERNAL_ERROR", "HIPBLAS_STATUS_INTERNAL_ERROR"),
    ("CUBLAS_STATUS_NOT_SUPPORTED", "HIPBLAS_STATUS_NOT_SUPPORTED"),
    ("CUBLAS_STATUS_ARCH_MISMATCH", "HIPBLAS_STATUS_ARCH_MISMATCH"),
    ("CUBLAS_FILL_MODE_LOWER", "HIPBLAS_FILL_MODE_LOWER"),
    ("CUBLAS_FILL_MODE_UPPER", "HIPBLAS_FILL_MODE_UPPER"),
    ("CUBLAS_DIAG_NON_UNIT", "HIPBLAS_DIAG_NON_UNIT"),
    ("CUBLAS_DIAG_UNIT", "HIPBLAS_DIAG_UNIT"),
    ("CUBLAS_SIDE_LEFT", "HIPBLAS_SIDE_LEFT"),
    ("CUBLAS_SIDE_RIGHT", "HIPBLAS_SIDE_RIGHT"),
    ("CUBLAS_POINTER_MODE_HOST", "HIPBLAS_POINTER_MODE_HOST"),
    ("CUBLAS_POINTER_MODE_DEVICE", "HIPBLAS_POINTER_MODE_DEVICE"),
    ("CUBLAS_ATOMICS_NOT_ALLOWED", "HIPBLAS_ATOMICS_NOT_ALLOWED"),
    ("CUBLAS_ATOMICS_ALLOWED", "HIPBLAS_ATOMICS_ALLOWED"),
    ("CUBLAS_DATA_FLOAT", "HIPBLAS_DATA_FLOAT"),
    ("CUBLAS_DATA_DOUBLE", "HIPBLAS_DATA_DOUBLE"),
    ("CUBLAS_DATA_HALF", "HIPBLAS_DATA_HALF"),
    ("CUBLAS_DATA_INT8", "HIPBLAS_DATA_INT8"),
    ("CUBLAS_GEMM_DEFAULT", "HIPBLAS_GEMM_DEFAULT"),
    ("CUBLAS_GEMM_DEFAULT_TENSOR_OP", "HIPBLAS_GEMM_DEFAULT"),
    ("cublasCreate", "hipblasCreate"),
    ("cublasDestroy", "hipblasDestroy"),
    ("cublasSetVector", "hipblasSetVector"),
    ("cublasGetVector", "hipblasGetVector"),
    ("cublasSetVectorAsync", "hipblasSetVectorAsync"),
    ("cublasGetVectorAsync", "hipblasGetVectorAsync"),
    ("cublasSetMatrix", "hipblasSetMatrix"),
    ("cublasGetMatrix", "hipblasGetMatrix"),
    ("cublasGetMatrixAsync", "hipblasGetMatrixAsync"),
    ("cublasSetMatrixAsync", "hipblasSetMatrixAsync"),
    ("cublasXerbla", "hipblasXerbla"),
    ("cublasSnrm2", "hipblasSnrm2"),
    ("cublasDnrm2", "hipblasDnrm2"),
    ("cublasScnrm2", "hipblasScnrm2"),
    ("cublasDznrm2", "hipblasDznrm2"),
    ("cublasNrm2Ex", "hipblasNrm2Ex"),
    ("cublasSdot", "hipblasSdot"),
    ("cublasSdotBatched", "hipblasSdotBatched"),
    ("cublasDdot", "hipblasDdot"),
    ("cublasDdotBatched", "hipblasDdotBatched"),
    ("cublasCdotu", "hipblasCdotu"),
    ("cublasCdotc", "hipblasCdotc"),
    ("cublasZdotu", "hipblasZdotu"),
    ("cublasZdotc", "hipblasZdotc"),
    ("cublasSscal", "hipblasSscal"),
    ("cublasSscalBatched", "hipblasSscalBatched"),
    ("cublasDscal", "hipblasDscal"),
    ("cublasDscalBatched", "hipblasDscalBatched"),
    ("cublasCscal", "hipblasCscal"),
    ("cublasCsscal", "hipblasCsscal"),
    ("cublasZscal", "hipblasZscal"),
    ("cublasZdscal", "hipblasZdscal"),
    ("cublasSaxpy", "hipblasSaxpy"),
    ("cublasSaxpyBatched", "hipblasSaxpyBatched"),
    ("cublasDaxpy", "hipblasDaxpy"),
    ("cublasCaxpy", "hipblasCaxpy"),
    ("cublasZaxpy", "hipblasZaxpy"),
    ("cublasScopy", "hipblasScopy"),
    ("cublasScopyBatched", "hipblasScopyBatched"),
    ("cublasDcopy", "hipblasDcopy"),
    ("cublasDcopyBatched", "hipblasDcopyBatched"),
    ("cublasCcopy", "hipblasCcopy"),
    ("cublasZcopy", "hipblasZcopy"),
    ("cublasSswap", "hipblasSswap"),
    ("cublasDswap", "hipblasDswap"),
    ("cublasCswap", "hipblasCswap"),
    ("cublasZswap", "hipblasZswap"),
    ("cublasIsamax", "hipblasIsamax"),
    ("cublasIdamax", "hipblasIdamax"),
    ("cublasIcamax", "hipblasIcamax"),
    ("cublasIzamax", "hipblasIzamax"),
    ("cublasIsamin", "hipblasIsamin"),
    ("cublasIdamin", "hipblasIdamin"),
    ("cublasIcamin", "hipblasIcamin"),
    ("cublasIzamin", "hipblasIzamin"),
    ("cublasSasum", "hipblasSasum"),
    ("cublasSasumBatched", "hipblasSasumBatched"),
    ("cublasDasum", "hipblasDasum"),
    ("cublasDasumBatched", "hipblasDasumBatched"),
    ("cublasScasum", "hipblasScasum"),
    ("cublasDzasum", "hipblasDzasum"),
    ("cublasSrot", "hipblasSrot"),
    ("cublasDrot", "hipblasDrot"),
    ("cublasCrot", "hipblasCrot"),
    ("cublasCsrot", "hipblasCsrot"),
    ("cublasZrot", "hipblasZrot"),
    ("cublasZdrot", "hipblasZdrot"),
    ("cublasSrotg", "hipblasSrotg"),
    ("cublasDrotg", "hipblasDrotg"),
    ("cublasCrotg", "hipblasCrotg"),
    ("cublasZrotg", "hipblasZrotg"),
    ("cublasSrotm", "hipblasSrotm"),
    ("cublasDrotm", "hipblasDrotm"),
    ("cublasSrotmg", "hipblasSrotmg"),
    ("cublasDrotmg", "hipblasDrotmg"),
    ("cublasSgemv", "hipblasSgemv"),
    ("cublasSgemvBatched", "hipblasSgemvBatched"),
    ("cublasDgemv", "hipblasDgemv"),
    ("cublasCgemv", "hipblasCgemv"),
    ("cublasZgemv", "hipblasZgemv"),
    ("cublasSgbmv", "hipblasSgbmv"),
    ("cublasDgbmv", "hipblasDgbmv"),
    ("cublasCgbmv", "hipblasCgbmv"),
    ("cublasZgbmv", "hipblasZgbmv"),
    ("cublasStrmv", "hipblasStrmv"),
    ("cublasDtrmv", "hipblasDtrmv"),
    ("cublasCtrmv", "hipblasCtrmv"),
    ("cublasZtrmv", "hipblasZtrmv"),
    ("cublasStbmv", "hipblasStbmv"),
    ("cublasDtbmv", "hipblasDtbmv"),
    ("cublasCtbmv", "hipblasCtbmv"),
    ("cublasZtbmv", "hipblasZtbmv"),
    ("cublasStpmv", "hipblasStpmv"),
    ("cublasDtpmv", "hipblasDtpmv"),
    ("cublasCtpmv", "hipblasCtpmv"),
    ("cublasZtpmv", "hipblasZtpmv"),
    ("cublasStrsv", "hipblasStrsv"),
    ("cublasDtrsv", "hipblasDtrsv"),
    ("cublasCtrsv", "hipblasCtrsv"),
    ("cublasZtrsv", "hipblasZtrsv"),
    ("cublasStpsv", "hipblasStpsv"),
    ("cublasDtpsv", "hipblasDtpsv"),
    ("cublasCtpsv", "hipblasCtpsv"),
    ("cublasZtpsv", "hipblasZtpsv"),
    ("cublasStbsv", "hipblasStbsv"),
    ("cublasDtbsv", "hipblasDtbsv"),
    ("cublasCtbsv", "hipblasCtbsv"),
    ("cublasZtbsv", "hipblasZtbsv"),
    ("cublasSsymv", "hipblasSsymv"),
    ("cublasDsymv", "hipblasDsymv"),
    ("cublasCsymv", "hipblasCsymv"),
    ("cublasZsymv", "hipblasZsymv"),
    ("cublasChemv", "hipblasChemv"),
    ("cublasZhemv", "hipblasZhemv"),
    ("cublasSsbmv", "hipblasSsbmv"),
    ("cublasDsbmv", "hipblasDsbmv"),
    ("cublasChbmv", "hipblasChbmv"),
    ("cublasZhbmv", "hipblasZhbmv"),
    ("cublasSspmv", "hipblasSspmv"),
    ("cublasDspmv", "hipblasDspmv"),
    ("cublasChpmv", "hipblasChpmv"),
    ("cublasZhpmv", "hipblasZhpmv"),
    ("cublasSger", "hipblasSger"),
    ("cublasDger", "hipblasDger"),
    ("cublasCgeru", "hipblasCgeru"),
    ("cublasCgerc", "hipblasCgerc"),
    ("cublasZgeru", "hipblasZgeru"),
    ("cublasZgerc", "hipblasZgerc"),
    ("cublasSsyr", "hipblasSsyr"),
    ("cublasDsyr", "hipblasDsyr"),
    ("cublasCher", "hipblasCher"),
    ("cublasZher", "hipblasZher"),
    ("cublasSspr", "hipblasSspr"),
    ("cublasDspr", "hipblasDspr"),
    ("cublasChpr", "hipblasChpr"),
    ("cublasZhpr", "hipblasZhpr"),
    ("cublasSsyr2", "hipblasSsyr2"),
    ("cublasDsyr2", "hipblasDsyr2"),
    ("cublasCher2", "hipblasCher2"),
    ("cublasZher2", "hipblasZher2"),
    ("cublasSspr2", "hipblasSspr2"),
    ("cublasDspr2", "hipblasDspr2"),
    ("cublasChpr2", "hipblasChpr2"),
    ("cublasZhpr2", "hipblasZhpr2"),
    ("cublasSgemmBatched", "hipblasSgemmBatched"),
    ("cublasDgemmBatched", "hipblasDgemmBatched"),
    ("cublasHgemmBatched", "hipblasHgemmBatched"),
    ("cublasSgemmStridedBatched", "hipblasSgemmStridedBatched"),
    ("cublasDgemmStridedBatched", "hipblasDgemmStridedBatched"),
    ("cublasHgemmStridedBatched", "hipblasHgemmStridedBatched"),
    ("cublasCgemmBatched", "hipblasCgemmBatched"),
    ("cublasCgemm3mBatched", "hipblasCgemm3mBatched"),
    ("cublasZgemmBatched", "hipblasZgemmBatched"),
    ("cublasCgemmStridedBatched", "hipblasCgemmStridedBatched"),
    ("cublasCgemm3mStridedBatched", "hipblasCgemm3mStridedBatched"),
    ("cublasZgemmStridedBatched", "hipblasZgemmStridedBatched"),
    ("cublasSgemm", "hipblasSgemm"),
    ("cublasDgemm", "hipblasDgemm"),
    ("cublasCgemm", "hipblasCgemm"),
    ("cublasZgemm", "hipblasZgemm"),
    ("cublasHgemm", "hipblasHgemm"),
    ("cublasSsyrk", "hipblasSsyrk"),
    ("cublasDsyrk", "hipblasDsyrk"),
    ("cublasCsyrk", "hipblasCsyrk"),
    ("cublasZsyrk", "hipblasZsyrk"),
    ("cublasCherk", "hipblasCherk"),
    ("cublasZherk", "hipblasZherk"),
    ("cublasSsyr2k", "hipblasSsyr2k"),
    ("cublasDsyr2k", "hipblasDsyr2k"),
    ("cublasCsyr2k", "hipblasCsyr2k"),
    ("cublasZsyr2k", "hipblasZsyr2k"),
    ("cublasSsyrkx", "hipblasSsyrkx"),
    ("cublasDsyrkx", "hipblasDsyrkx"),
    ("cublasCsyrkx", "hipblasCsyrkx"),
    ("cublasZsyrkx", "hipblasZsyrkx"),
    ("cublasCher2k", "hipblasCher2k"),
    ("cublasZher2k", "hipblasZher2k"),
    ("cublasCherkx", "hipblasCherkx"),
    ("cublasZherkx", "hipblasZherkx"),
    ("cublasSsymm", "hipblasSsymm"),
    ("cublasDsymm", "hipblasDsymm"),
    ("cublasCsymm", "hipblasCsymm"),
    ("cublasZsymm", "hipblasZsymm"),
    ("cublasChemm", "hipblasChemm"),
    ("cublasZhemm", "hipblasZhemm"),
    ("cublasStrsm", "hipblasStrsm"),
    ("cublasDtrsm", "hipblasDtrsm"),
    ("cublasCtrsm", "hipblasCtrsm"),
    ("cublasZtrsm", "hipblasZtrsm"),
    ("cublasStrsmBatched", "hipblasStrsmBatched"),
    ("cublasDtrsmBatched", "hipblasDtrsmBatched"),
    ("cublasCtrsmBatched", "hipblasCtrsmBatched"),
    ("cublasZtrsmBatched", "hipblasZtrsmBatched"),
    ("cublasStrmm", "hipblasStrmm"),
    ("cublasDtrmm", "hipblasDtrmm"),
    ("cublasCtrmm", "hipblasCtrmm"),
    ("cublasZtrmm", "hipblasZtrmm"),
    ("cublasSgeam", "hipblasSgeam"),
    ("cublasDgeam", "hipblasDgeam"),
    ("cublasCgeam", "hipblasCgeam"),
    ("cublasZgeam", "hipblasZgeam"),
    ("cublasSgetrfBatched", "hipblasSgetrfBatched"),
    ("cublasDgetrfBatched", "hipblasDgetrfBatched"),
    ("cublasCgetrfBatched", "hipblasCgetrfBatched"),
    ("cublasZgetrfBatched", "hipblasZgetrfBatched"),
    ("cublasSgetriBatched", "hipblasSgetriBatched"),
    ("cublasDgetriBatched", "hipblasDgetriBatched"),
    ("cublasCgetriBatched", "hipblasCgetriBatched"),
    ("cublasZgetriBatched", "hipblasZgetriBatched"),
    ("cublasSgetrsBatched", "hipblasSgetrsBatched"),
    ("cublasDgetrsBatched", "hipblasDgetrsBatched"),
    ("cublasCgetrsBatched", "hipblasCgetrsBatched"),
    ("cublasZgetrsBatched", "hipblasZgetrsBatched"),
    ("cublasSmatinvBatched", "hipblasSmatinvBatched"),
    ("cublasDmatinvBatched", "hipblasDmatinvBatched"),
    ("cublasCmatinvBatched", "hipblasCmatinvBatched"),
    ("cublasZmatinvBatched", "hipblasZmatinvBatched"),
    ("cublasSgeqrfBatched", "hipblasSgeqrfBatched"),
    ("cublasDgeqrfBatched", "hipblasDgeqrfBatched"),
    ("cublasCgeqrfBatched", "hipblasCgeqrfBatched"),
    ("cublasZgeqrfBatched", "hipblasZgeqrfBatched"),
    ("cublasSgelsBatched", "hipblasSgelsBatched"),
    ("cublasDgelsBatched", "hipblasDgelsBatched"),
    ("cublasCgelsBatched", "hipblasCgelsBatched"),
    ("cublasZgelsBatched", "hipblasZgelsBatched"),
    ("cublasSdgmm", "hipblasSdgmm"),
    ("cublasDdgmm", "hipblasDdgmm"),
    ("cublasCdgmm", "hipblasCdgmm"),
    ("cublasZdgmm", "hipblasZdgmm"),
    ("cublasStpttr", "hipblasStpttr"),
    ("cublasDtpttr", "hipblasDtpttr"),
    ("cublasCtpttr", "hipblasCtpttr"),
    ("cublasZtpttr", "hipblasZtpttr"),
    ("cublasStrttp", "hipblasStrttp"),
    ("cublasDtrttp", "hipblasDtrttp"),
    ("cublasCtrttp", "hipblasCtrttp"),
    ("cublasZtrttp", "hipblasZtrttp"),
    ("cublasCreate_v2", "hipblasCreate_v2"),
    ("cublasDestroy_v2", "hipblasDestroy_v2"),
    ("cublasGetVersion_v2", "hipblasGetVersion_v2"),
    ("cublasSetWorkspace", "hipblasSetWorkspace"),
    ("cublasSetStream", "hipblasSetStream"),
    ("cublasGetStream", "hipblasGetStream"),
    ("cublasSetStream_v2", "hipblasSetStream_v2"),
    ("cublasGetStream_v2", "hipblasGetStream_v2"),
    ("cublasGetPointerMode", "hipblasGetPointerMode"),
    ("cublasSetPointerMode", "hipblasSetPointerMode"),
    ("cublasGetPointerMode_v2", "hipblasGetPointerMode_v2"),
    ("cublasSetPointerMode_v2", "hipblasSetPointerMode_v2"),
    ("cublasSgemv_v2", "hipblasSgemv_v2"),
    ("cublasDgemv_v2", "hipblasDgemv_v2"),
    ("cublasCgemv_v2", "hipblasCgemv_v2"),
    ("cublasZgemv_v2", "hipblasZgemv_v2"),
    ("cublasSgbmv_v2", "hipblasSgbmv_v2"),
    ("cublasDgbmv_v2", "hipblasDgbmv_v2"),
    ("cublasCgbmv_v2", "hipblasCgbmv_v2"),
    ("cublasZgbmv_v2", "hipblasZgbmv_v2"),
    ("cublasStrmv_v2", "hipblasStrmv_v2"),
    ("cublasDtrmv_v2", "hipblasDtrmv_v2"),
    ("cublasCtrmv_v2", "hipblasCtrmv_v2"),
    ("cublasZtrmv_v2", "hipblasZtrmv_v2"),
    ("cublasStbmv_v2", "hipblasStbmv_v2"),
    ("cublasDtbmv_v2", "hipblasDtbmv_v2"),
    ("cublasCtbmv_v2", "hipblasCtbmv_v2"),
    ("cublasZtbmv_v2", "hipblasZtbmv_v2"),
    ("cublasStpmv_v2", "hipblasStpmv_v2"),
    ("cublasDtpmv_v2", "hipblasDtpmv_v2"),
    ("cublasCtpmv_v2", "hipblasCtpmv_v2"),
    ("cublasZtpmv_v2", "hipblasZtpmv_v2"),
    ("cublasStrsv_v2", "hipblasStrsv_v2"),
    ("cublasDtrsv_v2", "hipblasDtrsv_v2"),
    ("cublasCtrsv_v2", "hipblasCtrsv_v2"),
    ("cublasZtrsv_v2", "hipblasZtrsv_v2"),
    ("cublasStpsv_v2", "hipblasStpsv_v2"),
    ("cublasDtpsv_v2", "hipblasDtpsv_v2"),
    ("cublasCtpsv_v2", "hipblasCtpsv_v2"),
    ("cublasZtpsv_v2", "hipblasZtpsv_v2"),
    ("cublasStbsv_v2", "hipblasStbsv_v2"),
    ("cublasDtbsv_v2", "hipblasDtbsv_v2"),
    ("cublasCtbsv_v2", "hipblasCtbsv_v2"),
    ("cublasZtbsv_v2", "hipblasZtbsv_v2"),
    ("cublasSsymv_v2", "hipblasSsymv_v2"),
    ("cublasDsymv_v2", "hipblasDsymv_v2"),
    ("cublasCsymv_v2", "hipblasCsymv_v2"),
    ("cublasZsymv_v2", "hipblasZsymv_v2"),
    ("cublasChemv_v2", "hipblasChemv_v2"),
    ("cublasZhemv_v2", "hipblasZhemv_v2"),
    ("cublasSsbmv_v2", "hipblasSsbmv_v2"),
    ("cublasDsbmv_v2", "hipblasDsbmv_v2"),
    ("cublasChbmv_v2", "hipblasChbmv_v2"),
    ("cublasZhbmv_v2", "hipblasZhbmv_v2"),
    ("cublasSspmv_v2", "hipblasSspmv_v2"),
    ("cublasDspmv_v2", "hipblasDspmv_v2"),
    ("cublasChpmv_v2", "hipblasChpmv_v2"),
    ("cublasZhpmv_v2", "hipblasZhpmv_v2"),
    ("cublasSger_v2", "hipblasSger_v2"),
    ("cublasDger_v2", "hipblasDger_v2"),
    ("cublasCgeru_v2", "hipblasCgeru_v2"),
    ("cublasCgerc_v2", "hipblasCergc_v2"),
    ("cublasZgeru_v2", "hipblasZgeru_v2"),
    ("cublasZgerc_v2", "hipblasZgerc_v2"),
    ("cublasSsyr_v2", "hipblasSsyr_v2"),
    ("cublasDsyr_v2", "hipblasDsyr_v2"),
    ("cublasCsyr_v2", "hipblasCsyr_v2"),
    ("cublasZsyr_v2", "hipblasZsyr_v2"),
    ("cublasCher_v2", "hipblasCher_v2"),
    ("cublasZher_v2", "hipblasZher_v2"),
    ("cublasSspr_v2", "hipblasSspr_v2"),
    ("cublasDspr_v2", "hipblasDspr_v2"),
    ("cublasChpr_v2", "hipblasChpr_v2"),
    ("cublasZhpr_v2", "hipblasZhpr_v2"),
    ("cublasSsyr2_v2", "hipblasSsyr2_v2"),
    ("cublasDsyr2_v2", "hipblasDsyr2_v2"),
    ("cublasCsyr2_v2", "hipblasCsyr2_v2"),
    ("cublasZsyr2_v2", "hipblasZsyr2_v2"),
    ("cublasCher2_v2", "hipblasCher2_v2"),
    ("cublasZher2_v2", "hipblasZher2_v2"),
    ("cublasSspr2_v2", "hipblasSspr2_v2"),
    ("cublasDspr2_v2", "hipblasDspr2_v2"),
    ("cublasChpr2_v2", "hipblasChpr2_v2"),
    ("cublasZhpr2_v2", "hipblasZhpr2_v2"),
    ("cublasSgemm_v2", "hipblasSgemm_v2"),
    ("cublasDgemm_v2", "hipblasDgemm_v2"),
    ("cublasCgemm_v2", "hipblasCgemm_v2"),
    ("cublasCgemm3m", "hipblasCgemm3m"),
    ("cublasCgemm3mEx", "hipblasCgemm3mEx"),
    ("cublasZgemm_v2", "hipblasZgemm_v2"),
    ("cublasZgemm3m", "hipblasZgemm3m"),
    ("cublasSgemmEx", "hipblasSgemmEx"),
    ("cublasGemmEx", "hipblasGemmEx"),
    ("cublasGemmBatchedEx", "hipblasGemmBatchedEx"),
    ("cublasGemmStridedBatchedEx", "hipblasGemmStridedBatchedEx"),
    ("cublasCgemmEx", "hipblasCgemmEx"),
    ("cublasUint8gemmBias", "hipblasUint8gemmBias"),
    ("cublasSsyrk_v2", "hipblasSsyrk_v2"),
    ("cublasDsyrk_v2", "hipblasDsyrk_v2"),
    ("cublasCsyrk_v2", "hipblasCsyrk_v2"),
    ("cublasZsyrk_v2", "hipblasZsyrk_v2"),
    ("cublasCsyrkEx", "hipblasCsyrkEx"),
    ("cublasCsyrk3mEx", "hipblasCsyrk3mEx"),
    ("cublasCherk_v2", "hipblasCherk_v2"),
    ("cublasCherkEx", "hipblasCherkEx"),
    ("cublasCherk3mEx", "hipblasCherk3mEx"),
    ("cublasZherk_v2", "hipblasZherk_v2"),
    ("cublasSsyr2k_v2", "hipblasSsyr2k_v2"),
    ("cublasDsyr2k_v2", "hipblasDsyr2k_v2"),
    ("cublasCsyr2k_v2", "hipblasCsyr2k_v2"),
    ("cublasZsyr2k_v2", "hipblasZsyr2k_v2"),
    ("cublasCher2k_v2", "hipblasCher2k_v2"),
    ("cublasZher2k_v2", "hipblasZher2k_v2"),
    ("cublasSsymm_v2", "hipblasSsymm_v2"),
    ("cublasDsymm_v2", "hipblasDsymm_v2"),
    ("cublasCsymm_v2", "hipblasCsymm_v2"),
    ("cublasZsymm_v2", "hipblasZsymm_v2"),
    ("cublasChemm_v2", "hipblasChemm_v2"),
    ("cublasZhemm_v2", "hipblasZhemm_v2"),
    ("cublasStrsm_v2", "hipblasStrsm_v2"),
    ("cublasDtrsm_v2", "hipblasDtrsm_v2"),
    ("cublasCtrsm_v2", "hipblasCtrsm_v2"),
    ("cublasZtrsm_v2", "hipblasZtrsm_v2"),
    ("cublasStrmm_v2", "hipblasStrmm_v2"),
    ("cublasDtrmm_v2", "hipblasDtrmm_v2"),
    ("cublasCtrmm_v2", "hipblasCtrmm_v2"),
    ("cublasZtrmm_v2", "hipblasZtrmm_v2"),
    ("cublasSnrm2_v2", "hipblasSnrm2_v2"),
    ("cublasDnrm2_v2", "hipblasDnrm2_v2"),
    ("cublasScnrm2_v2", "hipblasScnrm2_v2"),
    ("cublasDznrm2_v2", "hipblasDznrm2_v2"),
    ("cublasDotEx", "hipblasDotEx"),
    ("cublasDotcEx", "hipblasDotcEx"),
    ("cublasSdot_v2", "hipblasSdot_v2"),
    ("cublasDdot_v2", "hipblasDdot_v2"),
    ("cublasCdotu_v2", "hipblasCdotu_v2"),
    ("cublasCdotc_v2", "hipblasCdotc_v2"),
    ("cublasZdotu_v2", "hipblasZdotu_v2"),
    ("cublasZdotc_v2", "hipblasZdotc_v2"),
    ("cublasScalEx", "hipblasScalEx"),
    ("cublasSscal_v2", "hipblasSscal_v2"),
    ("cublasDscal_v2", "hipblasDscal_v2"),
    ("cublasCscal_v2", "hipblasCscal_v2"),
    ("cublasCsscal_v2", "hipblasCsscal_v2"),
    ("cublasZscal_v2", "hipblasZcsal_v2"),
    ("cublasZdscal_v2", "hipblasZdscal_v2"),
    ("cublasAxpyEx", "hipblasAxpyEx"),
    ("cublasSaxpy_v2", "hipblasSaxpy_v2"),
    ("cublasDaxpy_v2", "hipblasDaxpy_v2"),
    ("cublasCaxpy_v2", "hipblasCaxpy_v2"),
    ("cublasZaxpy_v2", "hipblasZaxpy_v2"),
    ("cublasScopy_v2", "hipblasScopy_v2"),
    ("cublasDcopy_v2", "hipblasDcopy_v2"),
    ("cublasCcopy_v2", "hipblasCcopy_v2"),
    ("cublasZcopy_v2", "hipblasZcopy_v2"),
    ("cublasSswap_v2", "hipblasSswap_v2"),
    ("cublasDswap_v2", "hipblasDswap_v2"),
    ("cublasCswap_v2", "hipblasCswap_v2"),
    ("cublasZswap_v2", "hipblasZswap_v2"),
    ("cublasIsamax_v2", "hipblasIsamax_v2"),
    ("cublasIdamax_v2", "hipblasIdamax_v2"),
    ("cublasIcamax_v2", "hipblasIcamax_v2"),
    ("cublasIzamax_v2", "hipblasIzamax_v2"),
    ("cublasIsamin_v2", "hipblasIsamin_v2"),
    ("cublasIdamin_v2", "hipblasIdamin_v2"),
    ("cublasIcamin_v2", "hipblasIcamin_v2"),
    ("cublasIzamin_v2", "hipblasIzamin_v2"),
    ("cublasSasum_v2", "hipblasSasum_v2"),
    ("cublasDasum_v2", "hipblasDasum_v2"),
    ("cublasScasum_v2", "hipblasScasum_v2"),
    ("cublasDzasum_v2", "hipblasDzasum_v2"),
    ("cublasSrot_v2", "hipblasSrot_v2"),
    ("cublasDrot_v2", "hipblasDrot_v2"),
    ("cublasCrot_v2", "hipblasCrot_v2"),
    ("cublasCsrot_v2", "hipblasCsrot_v2"),
    ("cublasZrot_v2", "hipblasZrot_v2"),
    ("cublasZdrot_v2", "hipblasZdrot_v2"),
    ("cublasSrotg_v2", "hipblasSrotg_v2"),
    ("cublasDrotg_v2", "hipblasDrotg_v2"),
    ("cublasCrotg_v2", "hipblasCrotg_v2"),
    ("cublasZrotg_v2", "hipblasZrotg_v2"),
    ("cublasSrotm_v2", "hipblasSrotm_v2"),
    ("cublasDrotm_v2", "hipblasDrotm_v2"),
    ("cublasSrotmg_v2", "hipblasSrotmg_v2"),
    ("cublasDrotmg_v2", "hipblasDrotmg_v2"),
    ("cublasComputeType_t", "hipblasComputeType_t"),
    ("CUBLAS_COMPUTE_32I", "HIPBLAS_COMPUTE_32I"),
    ("CUBLAS_COMPUTE_32F", "HIPBLAS_COMPUTE_32F"),
    ("CUBLAS_COMPUTE_32F_FAST_TF32", "HIPBLAS_COMPUTE_32F_FAST_TF32"),
    ("CUBLAS_COMPUTE_64F", "HIPBLAS_COMPUTE_64F"),
    ("cublasLtEpilogue_t", "hipblasLtEpilogue_t"),
    ("CUBLASLT_EPILOGUE_DEFAULT", "HIPBLASLT_EPILOGUE_DEFAULT"),
    ("CUBLASLT_EPILOGUE_RELU", "HIPBLASLT_EPILOGUE_RELU"),
    ("CUBLASLT_EPILOGUE_BIAS", "HIPBLASLT_EPILOGUE_BIAS"),
    ("CUBLASLT_EPILOGUE_RELU_BIAS", "HIPBLASLT_EPILOGUE_RELU_BIAS"),
    ("CUBLASLT_EPILOGUE_GELU", "HIPBLASLT_EPILOGUE_GELU"),
    ("CUBLASLT_EPILOGUE_GELU_BIAS", "HIPBLASLT_EPILOGUE_GELU_BIAS"),
    ("cublasLtHandle_t", "hipblasLtHandle_t"),
    ("cublasLtMatmulDesc_t", "hipblasLtMatmulDesc_t"),
    ("cublasLtMatmulDescOpaque_t", "hipblasLtMatmulDescOpaque_t"),
    ("cublasLtMatmulDescAttributes_t", "hipblasLtMatmulDescAttributes_t"),
    ("CUBLASLT_MATMUL_DESC_TRANSA", "HIPBLASLT_MATMUL_DESC_TRANSA"),
    ("CUBLASLT_MATMUL_DESC_TRANSB", "HIPBLASLT_MATMUL_DESC_TRANSB"),
    ("CUBLASLT_MATMUL_DESC_EPILOGUE", "HIPBLASLT_MATMUL_DESC_EPILOGUE"),
    ("CUBLASLT_MATMUL_DESC_BIAS_POINTER", "HIPBLASLT_MATMUL_DESC_BIAS_POINTER"),
    ("CUBLASLT_MATMUL_DESC_A_SCALE_POINTER", "HIPBLASLT_MATMUL_DESC_A_SCALE_POINTER"),
    ("CUBLASLT_MATMUL_DESC_B_SCALE_POINTER", "HIPBLASLT_MATMUL_DESC_B_SCALE_POINTER"),
    ("CUBLASLT_MATMUL_DESC_D_SCALE_POINTER", "HIPBLASLT_MATMUL_DESC_D_SCALE_POINTER"),
    ("CUBLASLT_MATMUL_DESC_AMAX_D_POINTER", "HIPBLASLT_MATMUL_DESC_AMAX_D_POINTER"),
    ("CUBLASLT_MATMUL_DESC_BIAS_DATA_TYPE", "HIPBLASLT_MATMUL_DESC_BIAS_DATA_TYPE"),
    ("cublasLtMatrixLayout_t", "hipblasLtMatrixLayout_t"),
    ("cublasLtMatrixLayoutOpaque_t", "hipblasLtMatrixLayoutOpaque_t"),
    ("cublasLtMatrixLayoutAttribute_t", "hipblasLtMatrixLayoutAttribute_t"),
    ("cublasLtMatrixLayoutCreate", "hipblasLtMatrixLayoutCreate"),
    ("cublasLtMatrixLayoutDestroy", "hipblasLtMatrixLayoutDestroy"),
    ("cublasLtMatrixLayoutSetAttribute", "hipblasLtMatrixLayoutSetAttribute"),
    ("CUBLASLT_MATRIX_LAYOUT_BATCH_COUNT", "HIPBLASLT_MATRIX_LAYOUT_BATCH_COUNT"),
    ("CUBLASLT_MATRIX_LAYOUT_STRIDED_BATCH_OFFSET", "HIPBLASLT_MATRIX_LAYOUT_STRIDED_BATCH_OFFSET"),
    ("cublasLtMatmulPreference_t", "hipblasLtMatmulPreference_t"),
    ("cublasLtMatmulPreferenceOpaque_t", "hipblasLtMatmulPreferenceOpaque_t"),
    ("cublasLtMatmulPreferenceAttributes_t", "hipblasLtMatmulPreferenceAttributes_t"),
    ("CUBLASLT_MATMUL_PREF_SEARCH_MODE", "HIPBLASLT_MATMUL_PREF_SEARCH_MODE"),
    ("CUBLASLT_MATMUL_PREF_MAX_WORKSPACE_BYTES", "HIPBLASLT_MATMUL_PREF_MAX_WORKSPACE_BYTES"),
    ("cublasLtMatmulAlgo_t", "hipblasLtMatmulAlgo_t"),
    ("cublasLtMatmulHeuristicResult_t", "hipblasLtMatmulHeuristicResult_t"),
    ("cublasLtCreate", "hipblasLtCreate"),
    ("cublasLtDestroy", "hipblasLtDestroy"),
    ("cublasLtMatmulDescCreate", "hipblasLtMatmulDescCreate"),
    ("cublasLtMatmulDescDestroy", "hipblasLtMatmulDescDestroy"),
    ("cublasLtMatmulDescSetAttribute", "hipblasLtMatmulDescSetAttribute"),
    ("cublasLtMatmulPreferenceCreate", "hipblasLtMatmulPreferenceCreate"),
    ("cublasLtMatmulPreferenceDestroy", "hipblasLtMatmulPreferenceDestroy"),
    ("cublasLtMatmulPreferenceSetAttribute", "hipblasLtMatmulPreferenceSetAttribute"),
    ("cublasLtMatmulAlgoGetHeuristic", "hipblasLtMatmulAlgoGetHeuristic"),
    ("cublasLtMatmul", "hipblasLtMatmul"),
    ("CURAND_STATUS_SUCCESS", "HIPRAND_STATUS_SUCCESS"),
    ("CURAND_STATUS_VERSION_MISMATCH", "HIPRAND_STATUS_VERSION_MISMATCH"),
    ("CURAND_STATUS_NOT_INITIALIZED", "HIPRAND_STATUS_NOT_INITIALIZED"),
    ("CURAND_STATUS_ALLOCATION_FAILED", "HIPRAND_STATUS_ALLOCATION_FAILED"),
    ("CURAND_STATUS_TYPE_ERROR", "HIPRAND_STATUS_TYPE_ERROR"),
    ("CURAND_STATUS_OUT_OF_RANGE", "HIPRAND_STATUS_OUT_OF_RANGE"),
    ("CURAND_STATUS_LENGTH_NOT_MULTIPLE", "HIPRAND_STATUS_LENGTH_NOT_MULTIPLE"),
    ("CURAND_STATUS_DOUBLE_PRECISION_REQUIRED", "HIPRAND_STATUS_DOUBLE_PRECISION_REQUIRED"),
    ("CURAND_STATUS_LAUNCH_FAILURE", "HIPRAND_STATUS_LAUNCH_FAILURE"),
    ("CURAND_STATUS_PREEXISTING_FAILURE", "HIPRAND_STATUS_PREEXISTING_FAILURE"),
    ("CURAND_STATUS_INITIALIZATION_FAILED", "HIPRAND_STATUS_INITIALIZATION_FAILED"),
    ("CURAND_STATUS_ARCH_MISMATCH", "HIPRAND_STATUS_ARCH_MISMATCH"),
    ("CURAND_STATUS_INTERNAL_ERROR", "HIPRAND_STATUS_INTERNAL_ERROR"),
    ("CURAND_RNG_TEST", "HIPRAND_RNG_TEST"),
    ("mtgp32dc_params_fast_11213", "mtgp32dc_params_fast_11213"),
    ("CURAND_RNG_PSEUDO_DEFAULT", "HIPRAND_RNG_PSEUDO_DEFAULT"),
    ("CURAND_RNG_PSEUDO_XORWOW", "HIPRAND_RNG_PSEUDO_XORWOW"),
    ("CURAND_RNG_PSEUDO_MRG32K3A", "HIPRAND_RNG_PSEUDO_MRG32K3A"),
    ("CURAND_RNG_PSEUDO_MTGP32", "HIPRAND_RNG_PSEUDO_MTGP32"),
    ("CURAND_RNG_PSEUDO_MT19937", "HIPRAND_RNG_PSEUDO_MT19937"),
    ("CURAND_RNG_PSEUDO_PHILOX4_32_10", "HIPRAND_RNG_PSEUDO_PHILOX4_32_10"),
    ("CURAND_RNG_QUASI_DEFAULT", "HIPRAND_RNG_QUASI_DEFAULT"),
    ("CURAND_RNG_QUASI_SOBOL32", "HIPRAND_RNG_QUASI_SOBOL32"),
    ("CURAND_RNG_QUASI_SCRAMBLED_SOBOL32", "HIPRAND_RNG_QUASI_SCRAMBLED_SOBOL32"),
    ("CURAND_RNG_QUASI_SOBOL64", "HIPRAND_RNG_QUASI_SOBOL64"),
    ("CURAND_RNG_QUASI_SCRAMBLED_SOBOL64", "HIPRAND_RNG_QUASI_SCRAMBLED_SOBOL64"),
    ("curand_ORDERING_PSEUDO_BEST", "HIPRAND_ORDERING_PSEUDO_BEST"),
    ("curand_ORDERING_PSEUDO_DEFAULT", "HIPRAND_ORDERING_PSEUDO_DEFAULT"),
    ("curand_ORDERING_PSEUDO_SEEDED", "HIPRAND_ORDERING_PSEUDO_SEEDED"),
    ("curand_ORDERING_QUASI_DEFAULT", "HIPRAND_ORDERING_QUASI_DEFAULT"),
    ("curand_DIRECTION_VECTORS_32_JOEKUO6", "HIPRAND_DIRECTION_VECTORS_32_JOEKUO6"),
    ("curand_SCRAMBLED_DIRECTION_VECTORS_32_JOEKUO6", "HIPRAND_SCRAMBLED_DIRECTION_VECTORS_32_JOEKUO6"),
    ("curand_DIRECTION_VECTORS_64_JOEKUO6", "HIPRAND_DIRECTION_VECTORS_64_JOEKUO6"),
    ("curand_SCRAMBLED_DIRECTION_VECTORS_64_JOEKUO6", "HIPRAND_SCRAMBLED_DIRECTION_VECTORS_64_JOEKUO6"),
    ("curand_CHOOSE_BEST", "HIPRAND_CHOOSE_BEST"),
    ("curand_ITR", "HIPRAND_ITR"),
    ("curand_KNUTH", "HIPRAND_KNUTH"),
    ("curand_HITR", "HIPRAND_HITR"),
    ("curand_M1", "HIPRAND_M1"),
    ("curand_M2", "HIPRAND_M2"),
    ("curand_BINARY_SEARCH", "HIPRAND_BINARY_SEARCH"),
    ("curand_DISCRETE_GAUSS", "HIPRAND_DISCRETE_GAUSS"),
    ("curand_REJECTION", "HIPRAND_REJECTION"),
    ("curand_DEVICE_API", "HIPRAND_DEVICE_API"),
    ("curand_FAST_REJECTION", "HIPRAND_FAST_REJECTION"),
    ("curand_3RD", "HIPRAND_3RD"),
    ("curand_DEFINITION", "HIPRAND_DEFINITION"),
    ("curand_POISSON", "HIPRAND_POISSON"),
    ("curandCreateGenerator", "hiprandCreateGenerator"),
    ("curandCreateGeneratorHost", "hiprandCreateGeneratorHost"),
    ("curandCreatePoissonDistribution", "hiprandCreatePoissonDistribution"),
    ("curandDestroyDistribution", "hiprandDestroyDistribution"),
    ("curandDestroyGenerator", "hiprandDestroyGenerator"),
    ("curandGenerate", "hiprandGenerate"),
    ("curandGenerateLogNormal", "hiprandGenerateLogNormal"),
    ("curandGenerateLogNormalDouble", "hiprandGenerateLogNormalDouble"),
    ("curandGenerateLongLong", "hiprandGenerateLongLong"),
    ("curandGenerateNormal", "hiprandGenerateNormal"),
    ("curandGenerateNormalDouble", "hiprandGenerateNormalDouble"),
    ("curandGeneratePoisson", "hiprandGeneratePoisson"),
    ("curandGenerateSeeds", "hiprandGenerateSeeds"),
    ("curandGenerateUniform", "hiprandGenerateUniform"),
    ("curandGenerateUniformDouble", "hiprandGenerateUniformDouble"),
    ("curandGetDirectionVectors32", "hiprandGetDirectionVectors32"),
    ("curandGetDirectionVectors64", "hiprandGetDirectionVectors64"),
    ("curandGetProperty", "hiprandGetProperty"),
    ("curandGetScrambleConstants32", "hiprandGetScrambleConstants32"),
    ("curandGetScrambleConstants64", "hiprandGetScrambleConstants64"),
    ("curandGetVersion", "hiprandGetVersion"),
    ("curandSetGeneratorOffset", "hiprandSetGeneratorOffset"),
    ("curandSetGeneratorOrdering", "hiprandSetGeneratorOrdering"),
    ("curandSetPseudoRandomGeneratorSeed", "hiprandSetPseudoRandomGeneratorSeed"),
    ("curandSetQuasiRandomGeneratorDimensions", "hiprandSetQuasiRandomGeneratorDimensions"),
    ("curandSetStream", "hiprandSetStream"),
    ("curand", "hiprand"),
    ("curand4", "hiprand4"),
    ("curand_init", "hiprand_init"),
    ("curand_log_normal", "hiprand_log_normal"),
    ("curand_log_normal_double", "hiprand_log_normal_double"),
    ("curand_log_normal2", "hiprand_log_normal2"),
    ("curand_log_normal2_double", "hiprand_log_normal2_double"),
    ("curand_log_normal4", "hiprand_log_normal4"),
    ("curand_log_normal4_double", "hiprand_log_normal4_double"),
    ("curand_mtgp32_single", "hiprand_mtgp32_single"),
    ("curand_mtgp32_single_specific", "hiprand_mtgp32_single_specific"),
    ("curand_mtgp32_specific", "hiprand_mtgp32_specific"),
    ("curand_normal", "hiprand_normal"),
    ("curandMakeMTGP32Constants", "hiprandMakeMTGP32Constants"),
    ("curandMakeMTGP32KernelState", "hiprandMakeMTGP32KernelState"),
    ("curand_normal_double", "hiprand_normal_double"),
    ("curand_normal2", "hiprand_normal2"),
    ("curand_normal2_double", "hiprand_normal2_double"),
    ("curand_normal4", "hiprand_normal4"),
    ("curand_normal4_double", "hiprand_normal4_double"),
    ("curand_uniform", "hiprand_uniform"),
    ("curand_uniform_double", "hiprand_uniform_double"),
    ("curand_uniform2_double", "hiprand_uniform2_double"),
    ("curand_uniform4", "hiprand_uniform4"),
    ("curand_uniform4_double", "hiprand_uniform4_double"),
    ("curand_discrete", "hiprand_discrete"),
    ("curand_discrete4", "hiprand_discrete4"),
    ("curand_poisson", "hiprand_poisson"),
    ("curand_poisson4", "hiprand_poisson4"),
    ("curand_Philox4x32_10", "hiprand_Philox4x32_10"),
    ("mtgp32_kernel_params", "mtgp32_kernel_params_t"),
    ("CUFFT_FORWARD", "HIPFFT_FORWARD"),
    ("CUFFT_INVERSE", "HIPFFT_BACKWARD"),
    ("CUFFT_COMPATIBILITY_DEFAULT", "HIPFFT_COMPATIBILITY_DEFAULT"),
    ("cuComplex", "hipComplex"),
    ("cuDoubleComplex", "hipDoubleComplex"),
    ("cufftResult_t", "hipfftResult_t"),
    ("cufftResult", "hipfftResult"),
    ("CUFFT_SUCCESS", "HIPFFT_SUCCESS"),
    ("CUFFT_INVALID_PLAN", "HIPFFT_INVALID_PLAN"),
    ("CUFFT_ALLOC_FAILED", "HIPFFT_ALLOC_FAILED"),
    ("CUFFT_INVALID_TYPE", "HIPFFT_INVALID_TYPE"),
    ("CUFFT_INVALID_VALUE", "HIPFFT_INVALID_VALUE"),
    ("CUFFT_INTERNAL_ERROR", "HIPFFT_INTERNAL_ERROR"),
    ("CUFFT_EXEC_FAILED", "HIPFFT_EXEC_FAILED"),
    ("CUFFT_SETUP_FAILED", "HIPFFT_SETUP_FAILED"),
    ("CUFFT_INVALID_SIZE", "HIPFFT_INVALID_SIZE"),
    ("CUFFT_UNALIGNED_DATA", "HIPFFT_UNALIGNED_DATA"),
    ("CUFFT_INCOMPLETE_PARAMETER_LIST", "HIPFFT_INCOMPLETE_PARAMETER_LIST"),
    ("CUFFT_INVALID_DEVICE", "HIPFFT_INVALID_DEVICE"),
    ("CUFFT_PARSE_ERROR", "HIPFFT_PARSE_ERROR"),
    ("CUFFT_NO_WORKSPACE", "HIPFFT_NO_WORKSPACE"),
    ("CUFFT_NOT_IMPLEMENTED", "HIPFFT_NOT_IMPLEMENTED"),
    ("CUFFT_LICENSE_ERROR", "HIPFFT_LICENSE_ERROR"),
    ("CUFFT_NOT_SUPPORTED", "HIPFFT_NOT_SUPPORTED"),
    ("cufftType_t", "hipfftType_t"),
    ("cufftType", "hipfftType"),
    ("CUFFT_R2C", "HIPFFT_R2C"),
    ("CUFFT_C2R", "HIPFFT_C2R"),
    ("CUFFT_C2C", "HIPFFT_C2C"),
    ("CUFFT_D2Z", "HIPFFT_D2Z"),
    ("CUFFT_Z2D", "HIPFFT_Z2D"),
    ("CUFFT_Z2Z", "HIPFFT_Z2Z"),
    ("cufftCompatibility_t", "hipfftCompatibility_t"),
    ("cufftCompatibility", "hipfftCompatibility"),
    ("CUFFT_COMPATIBILITY_FFTW_PADDING", "HIPFFT_COMPATIBILITY_FFTW_PADDING"),
    ("cufftReal", "hipfftReal"),
    ("cufftDoubleReal", "hipfftDoubleReal"),
    ("cufftComplex", "hipfftComplex"),
    ("cufftDoubleComplex", "hipfftDoubleComplex"),
    ("cufftHandle", "hipfftHandle"),
    ("cufftPlan1d", "hipfftPlan1d"),
    ("cufftPlan2d", "hipfftPlan2d"),
    ("cufftPlan3d", "hipfftPlan3d"),
    ("cufftPlanMany", "hipfftPlanMany"),
    ("cufftMakePlan1d", "hipfftMakePlan1d"),
    ("cufftMakePlan2d", "hipfftMakePlan2d"),
    ("cufftMakePlan3d", "hipfftMakePlan3d"),
    ("cufftMakePlanMany", "hipfftMakePlanMany"),
    ("cufftMakePlanMany64", "hipfftMakePlanMany64"),
    ("cufftGetSizeMany64", "hipfftGetSizeMany64"),
    ("cufftEstimate1d", "hipfftEstimate1d"),
    ("cufftEstimate2d", "hipfftEstimate2d"),
    ("cufftEstimate3d", "hipfftEstimate3d"),
    ("cufftEstimateMany", "hipfftEstimateMany"),
    ("cufftCreate", "hipfftCreate"),
    ("cufftGetSize1d", "hipfftGetSize1d"),
    ("cufftGetSize2d", "hipfftGetSize2d"),
    ("cufftGetSize3d", "hipfftGetSize3d"),
    ("cufftGetSizeMany", "hipfftGetSizeMany"),
    ("cufftGetSize", "hipfftGetSize"),
    ("cufftSetWorkArea", "hipfftSetWorkArea"),
    ("cufftSetAutoAllocation", "hipfftSetAutoAllocation"),
    ("cufftXtExec", "hipfftXtExec"),
    ("cufftXtMakePlanMany", "hipfftXtMakePlanMany"),
    ("cufftExecC2C", "hipfftExecC2C"),
    ("cufftExecR2C", "hipfftExecR2C"),
    ("cufftExecC2R", "hipfftExecC2R"),
    ("cufftExecZ2Z", "hipfftExecZ2Z"),
    ("cufftExecD2Z", "hipfftExecD2Z"),
    ("cufftExecZ2D", "hipfftExecZ2D"),
    ("cufftSetStream", "hipfftSetStream"),
    ("cufftDestroy", "hipfftDestroy"),
    ("cufftGetVersion", "hipfftGetVersion"),
    ("cufftGetProperty", "hipfftGetProperty"),
    ("nvrtcResult", "hiprtcResult"),
    ("NVRTC_SUCCESS", "HIPRTC_SUCCESS"),
    ("NVRTC_ERROR_OUT_OF_MEMORY", "HIPRTC_ERROR_OUT_OF_MEMORY"),
    ("NVRTC_ERROR_PROGRAM_CREATION_FAILURE", "HIPRTC_ERROR_PROGRAM_CREATION_FAILURE"),
    ("NVRTC_ERROR_INVALID_INPUT", "HIPRTC_ERROR_INVALID_INPUT"),
    ("NVRTC_ERROR_INVALID_PROGRAM", "HIPRTC_ERROR_INVALID_PROGRAM"),
    ("NVRTC_ERROR_COMPILATION", "HIPRTC_ERROR_COMPILATION"),
    ("NVRTC_ERROR_BUILTIN_OPERATION_FAILURE", "HIPRTC_ERROR_BUILTIN_OPERATION_FAILURE"),
    ("NVRTC_ERROR_NO_NAME_EXPRESSIONS_AFTER_COMPILATION", "HIPRTC_ERROR_NO_NAME_EXPRESSIONS_AFTER_COMPILATION"),
    ("NVRTC_ERROR_NAME_EXPRESSION_NOT_VALID", "HIPRTC_ERROR_NAME_EXPRESSION_NOT_VALID"),
    ("NVRTC_ERROR_INTERNAL_ERROR", "HIPRTC_ERROR_INTERNAL_ERROR"),
    ("nvrtcGetErrorString", "hiprtcGetErrorString"),
    ("nvrtcVersion", "hiprtcVersion"),
    ("nvrtcProgram", "hiprtcProgram"),
    ("nvrtcAddNameExpression", "hiprtcAddNameExpression"),
    ("nvrtcCompileProgram", "hiprtcCompileProgram"),
    ("nvrtcCreateProgram", "hiprtcCreateProgram"),
    ("nvrtcDestroyProgram", "hiprtcDestroyProgram"),
    ("nvrtcGetLoweredName", "hiprtcGetLoweredName"),
    ("nvrtcGetProgramLog", "hiprtcGetProgramLog"),
    ("nvrtcGetProgramLogSize", "hiprtcGetProgramLogSize"),
    ("nvrtcGetPTX", "hiprtcGetCode"),
    ("nvrtcGetPTXSize", "hiprtcGetCodeSize"),
    ("nvrtcGetCUBIN", "hiprtcGetBitcode"),
    ("nvrtcGetCUBINSize", "hiprtcGetBitcodeSize"),
    ("thrust::cuda", "thrust::hip"),
    ("cub::", "hipcub::"),
    ("cub::ArgMax", "hipcub::ArgMax"),
    ("cub::ArgMin", "hipcub::ArgMin"),
    ("cub::BLOCK_SCAN_WARP_SCANS", "hipcub::BLOCK_SCAN_WARP_SCANS"),
    ("cub::BLOCK_REDUCE_WARP_REDUCTIONS", "hipcub::BLOCK_REDUCE_WARP_REDUCTIONS"),
    ("cub::BLOCK_STORE_WARP_TRANSPOSE", "hipcub::BLOCK_STORE_WARP_TRANSPOSE"),
    ("cub::BLOCK_LOAD_DIRECT", "hipcub::BLOCK_LOAD_DIRECT"),
    ("cub::BLOCK_STORE_DIRECT", "hipcub::BLOCK_STORE_DIRECT"),
    ("cub::BLOCK_REDUCE_RAKING_COMMUTATIVE_ONLY", "hipcub::BLOCK_REDUCE_RAKING_COMMUTATIVE_ONLY"),
    ("cub::BlockReduce", "hipcub::BlockReduce"),
    ("cub::BlockScan", "hipcub::BlockScan"),
    ("cub::BlockLoad", "hipcub::BlockLoad"),
    ("cub::BlockStore", "hipcub::BlockStore"),
    ("cub::BlockRakingLayout", "hipcub::BlockRakingLayout"),
    ("cub::BlockRadixSort", "hipcub::BlockRadixSort"),
    ("cub::Uninitialized", "hipcub::Uninitialized"),
    ("cub::RowMajorTid", "hipcub::RowMajorTid"),
    ("cub::CachingDeviceAllocator", "hipcub::CachingDeviceAllocator"),
    ("cub::CountingInputIterator", "hipcub::CountingInputIterator"),
    ("cub::DeviceRadixSort", "hipcub::DeviceRadixSort"),
    ("cub::DeviceReduce", "hipcub::DeviceReduce"),
    ("cub::DeviceRunLengthEncode", "hipcub::DeviceRunLengthEncode"),
    ("cub::DeviceScan", "hipcub::DeviceScan"),
    ("cub::DeviceSegmentedRadixSort", "hipcub::DeviceSegmentedRadixSort"),
    ("cub::DeviceSegmentedReduce", "hipcub::DeviceSegmentedReduce"),
    ("cub::DeviceSelect", "hipcub::DeviceSelect"),
    ("cub::FpLimits", "hipcub::FpLimits"),
    ("cub::KeyValuePair", "hipcub::KeyValuePair"),
    ("cub::Max", "hipcub::Max"),
    ("cub::Min", "hipcub::Min"),
    ("cub::Sum", "hipcub::Sum"),
    ("cub::Log2", "hipcub::Log2"),
    ("cub::LaneId", "hipcub::LaneId"),
    ("cub::WarpMask", "hipcub::WarpMask"),
    ("cub::ShuffleIndex", "hipcub::ShuffleIndex"),
    ("cub::ShuffleDown", "hipcub::ShuffleDown"),
    ("cub::ArgIndexInputIterator", "hipcub::ArgIndexInputIterator"),
    ("cub::TransformInputIterator", "hipcub::TransformInputIterator"),
    ("cub::WarpReduce", "hipcub::WarpReduce"),
    ("cub::CTA_SYNC", "hipcub::CTA_SYNC"),
    ("nvtxMark", "roctxMark"),
    ("nvtxMarkA", "roctxMarkA"),
    ("nvtxRangePushA", "roctxRangePushA"),
    ("nvtxRangePush", "roctxRangePush"),
    ("nvtxRangePop", "roctxRangePop"),
    ("nvtxRangeStartA", "roctxRangeStartA"),
    ("nvtxRangeStart", "roctxRangeStart"),
    ("nvtxRangeEnd", "roctxRangeStop"),
    ("nvtxRangeId_t", "int"),
    ("nvmlReturn_t", "rsmi_status_t"),
    ("NVML_SUCCESS", "RSMI_STATUS_SUCCESS"),
    ("NVML_P2P_CAPS_INDEX_READ", "RSMI_STATUS_SUCCESS"),
    ("NVML_P2P_STATUS_OK", "RSMI_STATUS_SUCCESS"),
    ("NVML_ERROR_INSUFFICIENT_SIZE", "RSMI_STATUS_INSUFFICIENT_SIZE"),
    ("nvmlDevice_t", "uint32_t"),
    ("nvmlGpuP2PStatus_t", "bool"),
    ("nvmlProcessInfo_t", "rsmi_process_info_t"),
    ("nvmlGpuP2PCapsIndex_t", "uint32_t"),
])

CUDA_SPECIAL_MAP = collections.OrderedDict([
    ("cusparseStatus_t", "hipsparseStatus_t"),
    ("cusparseHandle_t", "hipsparseHandle_t"),
    ("cuComplex", "hipComplex"),
    ("cuDoubleComplex", "hipDoubleComplex"),
    ("CUSPARSE_POINTER_MODE_HOST", "HIPSPARSE_POINTER_MODE_HOST"),
    ("cusparseOperation_t", "hipsparseOperation_t"),
    ("cusparseCreateMatDescr", "hipsparseCreateMatDescr"),
    ("cusparseCreate", "hipsparseCreate"),
    ("cusparseDestroyMatDescr", "hipsparseDestroyMatDescr"),
    ("cusparseDestroy", "hipsparseDestroy"),
    ("cusparseGetVersion", "hipsparseGetVersion"),
    ("cusparseXcoo2csr", "hipsparseXcoo2csr"),
    ("cusparseMatDescr_t", "hipsparseMatDescr_t"),
    ("cusparseDiagType_t", "hipsparseDiagType_t"),
    ("CUSPARSE_DIAG_TYPE_UNIT", "HIPSPARSE_DIAG_TYPE_UNIT"),
    ("CUSPARSE_DIAG_TYPE_NON_UNIT", "HIPSPARSE_DIAG_TYPE_NON_UNIT"),
    ("cusparseSetMatDiagType", "hipsparseSetMatDiagType"),
    ("cusparseFillMode_t", "hipsparseFillMode_t"),
    ("CUSPARSE_FILL_MODE_UPPER", "HIPSPARSE_FILL_MODE_UPPER"),
    ("CUSPARSE_FILL_MODE_LOWER", "HIPSPARSE_FILL_MODE_LOWER"),
    ("cusparseSetMatFillMode", "hipsparseSetMatFillMode"),
    ("cusparseDirection_t", "hipsparseDirection_t"),
    ("CUSPARSE_DIRECTION_ROW", "HIPSPARSE_DIRECTION_ROW"),
    ("CUSPARSE_DIRECTION_COLUMN", "HIPSPARSE_DIRECTION_COLUMN"),
    ("cusparseSolvePolicy_t", "hipsparseSolvePolicy_t"),
    ("CUSPARSE_SOLVE_POLICY_NO_LEVEL", "HIPSPARSE_SOLVE_POLICY_NO_LEVEL"),
    ("CUSPARSE_SOLVE_POLICY_USE_LEVEL", "HIPSPARSE_SOLVE_POLICY_USE_LEVEL"),
    ("cusparseCreateBsrsv2Info", "hipsparseCreateBsrsv2Info"),
    ("cusparseCreateBsrsm2Info", "hipsparseCreateBsrsm2Info"),
    ("cusparseDestroyBsrsv2Info", "hipsparseDestroyBsrsv2Info"),
    ("cusparseDestroyBsrsm2Info", "hipsparseDestroyBsrsm2Info"),
    ("cusparseSbsrmm", "hipsparseSbsrmm"),
    ("cusparseDbsrmm", "hipsparseDbsrmm"),
    ("cusparseCbsrmm", "hipsparseCbsrmm"),
    ("cusparseZbsrmm", "hipsparseZbsrmm"),
    ("cusparseSbsrmv", "hipsparseSbsrmv"),
    ("cusparseDbsrmv", "hipsparseDbsrmv"),
    ("cusparseCbsrmv", "hipsparseCbsrmv"),
    ("cusparseZbsrmv", "hipsparseZbsrmv"),
    ("cusparseSbsrsv2_bufferSize", "hipsparseSbsrsv2_bufferSize"),
    ("cusparseDbsrsv2_bufferSize", "hipsparseDbsrsv2_bufferSize"),
    ("cusparseCbsrsv2_bufferSize", "hipsparseCbsrsv2_bufferSize"),
    ("cusparseZbsrsv2_bufferSize", "hipsparseZbsrsv2_bufferSize"),
    ("cusparseSbsrsv2_analysis", "hipsparseSbsrsv2_analysis"),
    ("cusparseDbsrsv2_analysis", "hipsparseDbsrsv2_analysis"),
    ("cusparseCbsrsv2_analysis", "hipsparseCbsrsv2_analysis"),
    ("cusparseZbsrsv2_analysis", "hipsparseZbsrsv2_analysis"),
    ("cusparseSbsrsv2_solve", "hipsparseSbsrsv2_solve"),
    ("cusparseDbsrsv2_solve", "hipsparseDbsrsv2_solve"),
    ("cusparseCbsrsv2_solve", "hipsparseCbsrsv2_solve"),
    ("cusparseZbsrsv2_solve", "hipsparseZbsrsv2_solve"),
    ("cusparseSbsrsm2_bufferSize", "hipsparseSbsrsm2_bufferSize"),
    ("cusparseDbsrsm2_bufferSize", "hipsparseDbsrsm2_bufferSize"),
    ("cusparseCbsrsm2_bufferSize", "hipsparseCbsrsm2_bufferSize"),
    ("cusparseZbsrsm2_bufferSize", "hipsparseZbsrsm2_bufferSize"),
    ("cusparseSbsrsm2_analysis", "hipsparseSbsrsm2_analysis"),
    ("cusparseDbsrsm2_analysis", "hipsparseDbsrsm2_analysis"),
    ("cusparseCbsrsm2_analysis", "hipsparseCbsrsm2_analysis"),
    ("cusparseZbsrsm2_analysis", "hipsparseZbsrsm2_analysis"),
    ("cusparseSbsrsm2_solve", "hipsparseSbsrsm2_solve"),
    ("cusparseDbsrsm2_solve", "hipsparseDbsrsm2_solve"),
    ("cusparseCbsrsm2_solve", "hipsparseCbsrsm2_solve"),
    ("cusparseZbsrsm2_solve", "hipsparseZbsrsm2_solve"),
    ("cusparseScsrmm2", "hipsparseScsrmm2"),
    ("cusparseDcsrmm2", "hipsparseDcsrmm2"),
    ("cusparseCcsrmm2", "hipsparseCcsrmm2"),
    ("cusparseZcsrmm2", "hipsparseZcsrmm2"),
    ("cusparseScsrmm", "hipsparseScsrmm"),
    ("cusparseDcsrmm", "hipsparseDcsrmm"),
    ("cusparseCcsrmm", "hipsparseCcsrmm"),
    ("cusparseZcsrmm", "hipsparseZcsrmm"),
    ("cusparseXcsrsort_bufferSizeExt", "hipsparseXcsrsort_bufferSizeExt"),
    ("cusparseCreateCsrgemm2Info", "hipsparseCreateCsrgemm2Info"),
    ("cusparseDestroyCsrgemm2Info", "hipsparseDestroyCsrgemm2Info"),
    ("cusparseXcsrgemm2Nnz", "hipsparseXcsrgemm2Nnz"),
    ("cusparseDcsrgemm2_bufferSizeExt", "hipsparseDcsrgemm2_bufferSizeExt"),
    ("cusparseScsrgemm2_bufferSizeExt", "hipsparseScsrgemm2_bufferSizeExt"),
    ("cusparseDcsrgemm2", "hipsparseDcsrgemm2"),
    ("cusparseScsrgemm2", "hipsparseScsrgemm2"),
    ("cusparseScsrgemm", "hipsparseScsrgemm"),
    ("cusparseDcsrgemm", "hipsparseDcsrgemm"),
    ("cusparseCcsrgemm", "hipsparseCcsrgemm"),
    ("cusparseZcsrgemm", "hipsparseZcsrgemm"),
    ("cusparseSetPointerMode", "hipsparseSetPointerMode"),
    ("cusparseXcsrgeam2Nnz", "hipsparseXcsrgeam2Nnz"),
    ("cusparseScsrgeam", "hipsparseScsrgeam"),
    ("cusparseDcsrgeam", "hipsparseDcsrgeam"),
    ("cusparseCcsrgeam", "hipsparseCcsrgeam"),
    ("cusparseZcsrgeam", "hipsparseZcsrgeam"),
    ("cusparseScsrgeam2_bufferSizeExt", "hipsparseScsrgeam2_bufferSizeExt"),
    ("cusparseDcsrgeam2_bufferSizeExt", "hipsparseDcsrgeam2_bufferSizeExt"),
    ("cusparseCcsrgeam2_bufferSizeExt", "hipsparseCcsrgeam2_bufferSizeExt"),
    ("cusparseZcsrgeam2_bufferSizeExt", "hipsparseZcsrgeam2_bufferSizeExt"),
    ("cusparseScsrgeam2", "hipsparseScsrgeam2"),
    ("cusparseDcsrgeam2", "hipsparseDcsrgeam2"),
    ("cusparseCcsrgeam2", "hipsparseCcsrgeam2"),
    ("cusparseZcsrgeam2", "hipsparseZcsrgeam2"),
    ("cusparseXcsrsort", "hipsparseXcsrsort"),
    ("cusparseXbsrsm2_zeroPivot", "hipsparseXbsrsm2_zeroPivot"),
    ("cusparseXbsrsv2_zeroPivot", "hipsparseXbsrsv2_zeroPivot"),
    ("cusparseXcoosort_bufferSizeExt", "hipsparseXcoosort_bufferSizeExt"),
    ("cusparseXcoosortByRow", "hipsparseXcoosortByRow"),
    ("cusparseSetStream", "hipsparseSetStream"),
    ("cusparseGetStream", "hipsparseGetStream"),
    ("cusparseCreateIdentityPermutation", "hipsparseCreateIdentityPermutation"),
    ("cusparseSetMatIndexBase", "hipsparseSetMatIndexBase"),
    ("cusparseSetMatType", "hipsparseSetMatType"),
    ("cusparseSgthr", "hipsparseSgthr"),
    ("cusparseDgthr", "hipsparseDgthr"),
    ("cusparseCgthr", "hipsparseCgthr"),
    ("cusparseZgthr", "hipsparseZgthr"),
    ("cusparseScsrmv", "hipsparseScsrmv"),
    ("cusparseDcsrmv", "hipsparseDcsrmv"),
    ("cusparseCcsrmv", "hipsparseCcsrmv"),
    ("cusparseZcsrmv", "hipsparseZcsrmv"),
    ("cusparseSpMV", "hipsparseSpMV"),
    ("cusparseSpMV_bufferSize", "hipsparseSpMV_bufferSize"),
    ("cusparseSpMM", "hipsparseSpMM"),
    ("cusparseSpMM_bufferSize", "hipsparseSpMM_bufferSize"),
    ("cusparseCreateCsrsv2Info", "hipsparseCreateCsrsv2Info"),
    ("cusparseDestroyCsrsv2Info", "hipsparseDestroyCsrsv2Info"),
    ("cusparseScsrsv2_bufferSize", "hipsparseScsrsv2_bufferSize"),
    ("cusparseDcsrsv2_bufferSize", "hipsparseDcsrsv2_bufferSize"),
    ("cusparseCcsrsv2_bufferSize", "hipsparseCcsrsv2_bufferSize"),
    ("cusparseZcsrsv2_bufferSize", "hipsparseZcsrsv2_bufferSize"),
    ("cusparseScsrsv2_analysis", "hipsparseScsrsv2_analysis"),
    ("cusparseDcsrsv2_analysis", "hipsparseDcsrsv2_analysis"),
    ("cusparseCcsrsv2_analysis", "hipsparseCcsrsv2_analysis"),
    ("cusparseZcsrsv2_analysis", "hipsparseZcsrsv2_analysis"),
    ("cusparseScsrsv2_solve", "hipsparseScsrsv2_solve"),
    ("cusparseDcsrsv2_solve", "hipsparseDcsrsv2_solve"),
    ("cusparseCcsrsv2_solve", "hipsparseCcsrsv2_solve"),
    ("cusparseZcsrsv2_solve", "hipsparseZcsrsv2_solve"),
    ("cusparseXcsrsv2_zeroPivot", "hipsparseXcsrsv2_zeroPivot"),
    ("cusparseCreateCsrsm2Info", "hipsparseCreateCsrsm2Info"),
    ("cusparseDestroyCsrsm2Info", "hipsparseDestroyCsrsm2Info"),
    ("cusparseScsrsm2_bufferSizeExt", "hipsparseScsrsm2_bufferSizeExt"),
    ("cusparseDcsrsm2_bufferSizeExt", "hipsparseDcsrsm2_bufferSizeExt"),
    ("cusparseCcsrsm2_bufferSizeExt", "hipsparseCcsrsm2_bufferSizeExt"),
    ("cusparseZcsrsm2_bufferSizeExt", "hipsparseZcsrsm2_bufferSizeExt"),
    ("cusparseScsrsm2_analysis", "hipsparseScsrsm2_analysis"),
    ("cusparseDcsrsm2_analysis", "hipsparseDcsrsm2_analysis"),
    ("cusparseCcsrsm2_analysis", "hipsparseCcsrsm2_analysis"),
    ("cusparseZcsrsm2_analysis", "hipsparseZcsrsm2_analysis"),
    ("cusparseScsrsm2_solve", "hipsparseScsrsm2_solve"),
    ("cusparseDcsrsm2_solve", "hipsparseDcsrsm2_solve"),
    ("cusparseCcsrsm2_solve", "hipsparseCcsrsm2_solve"),
    ("cusparseZcsrsm2_solve", "hipsparseZcsrsm2_solve"),
    ("cusparseXcsrsm2_zeroPivot", "hipsparseXcsrsm2_zeroPivot"),
    ("cusparseXcsrgeamNnz", "hipsparseXcsrgeamNnz"),
    ("cusparseXcsrgemmNnz", "hipsparseXcsrgemmNnz"),
    ("cusparseCcsrgemm2_bufferSizeExt", "hipsparseCcsrgemm2_bufferSizeExt"),
    ("cusparseZcsrgemm2_bufferSizeExt", "hipsparseZcsrgemm2_bufferSizeExt"),
    ("cusparseCcsrgemm2", "hipsparseCcsrgemm2"),
    ("cusparseZcsrgemm2", "hipsparseZcsrgemm2"),
    ("cusparseScsc2dense", "hipsparseScsc2dense"),
    ("cusparseDcsc2dense", "hipsparseDcsc2dense"),
    ("cusparseCcsc2dense", "hipsparseCcsc2dense"),
    ("cusparseZcsc2dense", "hipsparseZcsc2dense"),
    ("cusparseXcsr2coo", "hipsparseXcsr2coo"),
    ("cusparseScsr2csc", "hipsparseScsr2csc"),
    ("cusparseDcsr2csc", "hipsparseDcsr2csc"),
    ("cusparseCcsr2csc", "hipsparseCcsr2csc"),
    ("cusparseZcsr2csc", "hipsparseZcsr2csc"),
    ("cusparseScsr2dense", "hipsparseScsr2dense"),
    ("cusparseDcsr2dense", "hipsparseDcsr2dense"),
    ("cusparseCcsr2dense", "hipsparseCcsr2dense"),
    ("cusparseZcsr2dense", "hipsparseZcsr2dense"),
    ("cusparseSnnz_compress", "hipsparseSnnz_compress"),
    ("cusparseDnnz_compress", "hipsparseDnnz_compress"),
    ("cusparseCnnz_compress", "hipsparseCnnz_compress"),
    ("cusparseZnnz_compress", "hipsparseZnnz_compress"),
    ("cusparseScsr2csr_compress", "hipsparseScsr2csr_compress"),
    ("cusparseDcsr2csr_compress", "hipsparseDcsr2csr_compress"),
    ("cusparseCcsr2csr_compress", "hipsparseCcsr2csr_compress"),
    ("cusparseZcsr2csr_compress", "hipsparseZcsr2csr_compress"),
    ("cusparseSdense2csc", "hipsparseSdense2csc"),
    ("cusparseDdense2csc", "hipsparseDdense2csc"),
    ("cusparseCdense2csc", "hipsparseCdense2csc"),
    ("cusparseZdense2csc", "hipsparseZdense2csc"),
    ("cusparseSdense2csr", "hipsparseSdense2csr"),
    ("cusparseDdense2csr", "hipsparseDdense2csr"),
    ("cusparseCdense2csr", "hipsparseCdense2csr"),
    ("cusparseZdense2csr", "hipsparseZdense2csr"),
    ("cusparseSnnz", "hipsparseSnnz"),
    ("cusparseDnnz", "hipsparseDnnz"),
    ("cusparseCnnz", "hipsparseCnnz"),
    ("cusparseZnnz", "hipsparseZnnz"),
    ("cusparseXcoosortByColumn", "hipsparseXcoosortByColumn"),
    ("cusparseXcscsort_bufferSizeExt", "hipsparseXcscsort_bufferSizeExt"),
    ("cusparseXcscsort", "hipsparseXcscsort"),
    ("cusparseCreateCsrilu02Info", "hipsparseCreateCsrilu02Info"),
    ("cusparseDestroyCsrilu02Info", "hipsparseDestroyCsrilu02Info"),
    ("cusparseCreateBsrilu02Info", "hipsparseCreateBsrilu02Info"),
    ("cusparseDestroyBsrilu02Info", "hipsparseDestroyBsrilu02Info"),
    ("cusparseCreateCsric02Info", "hipsparseCreateCsric02Info"),
    ("cusparseDestroyCsric02Info", "hipsparseDestroyCsric02Info"),
    ("cusparseCreateBsric02Info", "hipsparseCreateBsric02Info"),
    ("cusparseDestroyBsric02Info", "hipsparseDestroyBsric02Info"),
    ("cusparseScsrilu02_numericBoost", "hipsparseScsrilu02_numericBoost"),
    ("cusparseDcsrilu02_numericBoost", "hipsparseDcsrilu02_numericBoost"),
    ("cusparseCcsrilu02_numericBoost", "hipsparseCcsrilu02_numericBoost"),
    ("cusparseZcsrilu02_numericBoost", "hipsparseZcsrilu02_numericBoost"),
    ("cusparseXcsrilu02_zeroPivot", "hipsparseXcsrilu02_zeroPivot"),
    ("cusparseScsrilu02_bufferSize", "hipsparseScsrilu02_bufferSize"),
    ("cusparseDcsrilu02_bufferSize", "hipsparseDcsrilu02_bufferSize"),
    ("cusparseCcsrilu02_bufferSize", "hipsparseCcsrilu02_bufferSize"),
    ("cusparseZcsrilu02_bufferSize", "hipsparseZcsrilu02_bufferSize"),
    ("cusparseScsrilu02_analysis", "hipsparseScsrilu02_analysis"),
    ("cusparseDcsrilu02_analysis", "hipsparseDcsrilu02_analysis"),
    ("cusparseCcsrilu02_analysis", "hipsparseCcsrilu02_analysis"),
    ("cusparseZcsrilu02_analysis", "hipsparseZcsrilu02_analysis"),
    ("cusparseScsrilu02", "hipsparseScsrilu02"),
    ("cusparseDcsrilu02", "hipsparseDcsrilu02"),
    ("cusparseCcsrilu02", "hipsparseCcsrilu02"),
    ("cusparseZcsrilu02", "hipsparseZcsrilu02"),
    ("cusparseSbsrilu02_numericBoost", "hipsparseSbsrilu02_numericBoost"),
    ("cusparseDbsrilu02_numericBoost", "hipsparseDbsrilu02_numericBoost"),
    ("cusparseCbsrilu02_numericBoost", "hipsparseCbsrilu02_numericBoost"),
    ("cusparseZbsrilu02_numericBoost", "hipsparseZbsrilu02_numericBoost"),
    ("cusparseXbsrilu02_zeroPivot", "hipsparseXbsrilu02_zeroPivot"),
    ("cusparseSbsrilu02_bufferSize", "hipsparseSbsrilu02_bufferSize"),
    ("cusparseDbsrilu02_bufferSize", "hipsparseDbsrilu02_bufferSize"),
    ("cusparseCbsrilu02_bufferSize", "hipsparseCbsrilu02_bufferSize"),
    ("cusparseZbsrilu02_bufferSize", "hipsparseZbsrilu02_bufferSize"),
    ("cusparseSbsrilu02_analysis", "hipsparseSbsrilu02_analysis"),
    ("cusparseDbsrilu02_analysis", "hipsparseDbsrilu02_analysis"),
    ("cusparseCbsrilu02_analysis", "hipsparseCbsrilu02_analysis"),
    ("cusparseZbsrilu02_analysis", "hipsparseZbsrilu02_analysis"),
    ("cusparseSbsrilu02", "hipsparseSbsrilu02"),
    ("cusparseDbsrilu02", "hipsparseDbsrilu02"),
    ("cusparseCbsrilu02", "hipsparseCbsrilu02"),
    ("cusparseZbsrilu02", "hipsparseZbsrilu02"),
    ("cusparseXcsric02_zeroPivot", "hipsparseXcsric02_zeroPivot"),
    ("cusparseScsric02_bufferSize", "hipsparseScsric02_bufferSize"),
    ("cusparseDcsric02_bufferSize", "hipsparseDcsric02_bufferSize"),
    ("cusparseCcsric02_bufferSize", "hipsparseCcsric02_bufferSize"),
    ("cusparseZcsric02_bufferSize", "hipsparseZcsric02_bufferSize"),
    ("cusparseScsric02_analysis", "hipsparseScsric02_analysis"),
    ("cusparseDcsric02_analysis", "hipsparseDcsric02_analysis"),
    ("cusparseCcsric02_analysis", "hipsparseCcsric02_analysis"),
    ("cusparseZcsric02_analysis", "hipsparseZcsric02_analysis"),
    ("cusparseScsric02", "hipsparseScsric02"),
    ("cusparseDcsric02", "hipsparseDcsric02"),
    ("cusparseCcsric02", "hipsparseCcsric02"),
    ("cusparseZcsric02", "hipsparseZcsric02"),
    ("cusparseXbsric02_zeroPivot", "hipsparseXbsric02_zeroPivot"),
    ("cusparseSbsric02_bufferSize", "hipsparseSbsric02_bufferSize"),
    ("cusparseDbsric02_bufferSize", "hipsparseDbsric02_bufferSize"),
    ("cusparseCbsric02_bufferSize", "hipsparseCbsric02_bufferSize"),
    ("cusparseZbsric02_bufferSize", "hipsparseZbsric02_bufferSize"),
    ("cusparseSbsric02_analysis", "hipsparseSbsric02_analysis"),
    ("cusparseDbsric02_analysis", "hipsparseDbsric02_analysis"),
    ("cusparseCbsric02_analysis", "hipsparseCbsric02_analysis"),
    ("cusparseZbsric02_analysis", "hipsparseZbsric02_analysis"),
    ("cusparseSbsric02", "hipsparseSbsric02"),
    ("cusparseDbsric02", "hipsparseDbsric02"),
    ("cusparseCbsric02", "hipsparseCbsric02"),
    ("cusparseZbsric02", "hipsparseZbsric02"),
    ("cusparseSgtsv2_bufferSizeExt", "hipsparseSgtsv2_bufferSizeExt"),
    ("cusparseDgtsv2_bufferSizeExt", "hipsparseDgtsv2_bufferSizeExt"),
    ("cusparseCgtsv2_bufferSizeExt", "hipsparseCgtsv2_bufferSizeExt"),
    ("cusparseZgtsv2_bufferSizeExt", "hipsparseZgtsv2_bufferSizeExt"),
    ("cusparseSgtsv2", "hipsparseSgtsv2"),
    ("cusparseDgtsv2", "hipsparseDgtsv2"),
    ("cusparseCgtsv2", "hipsparseCgtsv2"),
    ("cusparseZgtsv2", "hipsparseZgtsv2"),
    ("cusparseSgtsv2_nopivot_bufferSizeExt", "hipsparseSgtsv2_nopivot_bufferSizeExt"),
    ("cusparseDgtsv2_nopivot_bufferSizeExt", "hipsparseDgtsv2_nopivot_bufferSizeExt"),
    ("cusparseCgtsv2_nopivot_bufferSizeExt", "hipsparseCgtsv2_nopivot_bufferSizeExt"),
    ("cusparseZgtsv2_nopivot_bufferSizeExt", "hipsparseZgtsv2_nopivot_bufferSizeExt"),
    ("cusparseSgtsv2_nopivot", "hipsparseSgtsv2_nopivot"),
    ("cusparseDgtsv2_nopivot", "hipsparseDgtsv2_nopivot"),
    ("cusparseCgtsv2_nopivot", "hipsparseCgtsv2_nopivot"),
    ("cusparseZgtsv2_nopivot", "hipsparseZgtsv2_nopivot"),
    ("cusparseSgtsv2StridedBatch_bufferSizeExt", "hipsparseSgtsv2StridedBatch_bufferSizeExt"),
    ("cusparseDgtsv2StridedBatch_bufferSizeExt", "hipsparseDgtsv2StridedBatch_bufferSizeExt"),
    ("cusparseCgtsv2StridedBatch_bufferSizeExt", "hipsparseCgtsv2StridedBatch_bufferSizeExt"),
    ("cusparseZgtsv2StridedBatch_bufferSizeExt", "hipsparseZgtsv2StridedBatch_bufferSizeExt"),
    ("cusparseSgtsv2StridedBatch", "hipsparseSgtsv2StridedBatch"),
    ("cusparseDgtsv2StridedBatch", "hipsparseDgtsv2StridedBatch"),
    ("cusparseCgtsv2StridedBatch", "hipsparseCgtsv2StridedBatch"),
    ("cusparseZgtsv2StridedBatch", "hipsparseZgtsv2StridedBatch"),
    ("cusparseSgtsvInterleavedBatch_bufferSizeExt", "hipsparseSgtsvInterleavedBatch_bufferSizeExt"),
    ("cusparseDgtsvInterleavedBatch_bufferSizeExt", "hipsparseDgtsvInterleavedBatch_bufferSizeExt"),
    ("cusparseCgtsvInterleavedBatch_bufferSizeExt", "hipsparseCgtsvInterleavedBatch_bufferSizeExt"),
    ("cusparseZgtsvInterleavedBatch_bufferSizeExt", "hipsparseZgtsvInterleavedBatch_bufferSizeExt"),
    ("cusparseSgtsvInterleavedBatch", "hipsparseSgtsvInterleavedBatch"),
    ("cusparseDgtsvInterleavedBatch", "hipsparseDgtsvInterleavedBatch"),
    ("cusparseCgtsvInterleavedBatch", "hipsparseCgtsvInterleavedBatch"),
    ("cusparseZgtsvInterleavedBatch", "hipsparseZgtsvInterleavedBatch"),
    ("cusparseSgpsvInterleavedBatch_bufferSizeExt", "hipsparseSgpsvInterleavedBatch_bufferSizeExt"),
    ("cusparseDgpsvInterleavedBatch_bufferSizeExt", "hipsparseDgpsvInterleavedBatch_bufferSizeExt"),
    ("cusparseCgpsvInterleavedBatch_bufferSizeExt", "hipsparseCgpsvInterleavedBatch_bufferSizeExt"),
    ("cusparseZgpsvInterleavedBatch_bufferSizeExt", "hipsparseZgpsvInterleavedBatch_bufferSizeExt"),
    ("cusparseSgpsvInterleavedBatch", "hipsparseSgpsvInterleavedBatch"),
    ("cusparseDgpsvInterleavedBatch", "hipsparseDgpsvInterleavedBatch"),
    ("cusparseCgpsvInterleavedBatch", "hipsparseCgpsvInterleavedBatch"),
    ("cusparseZgpsvInterleavedBatch", "hipsparseZgpsvInterleavedBatch"),
    ("cusparseCreateSpVec", "hipsparseCreateSpVec"),
    ("cusparseDestroySpVec", "hipsparseDestroySpVec"),
    ("cusparseSpVecGet", "hipsparseSpVecGet"),
    ("cusparseSpVecGetIndexBase", "hipsparseSpVecGetIndexBase"),
    ("cusparseSpVecGetValues", "hipsparseSpVecGetValues"),
    ("cusparseSpVecSetValues", "hipsparseSpVecSetValues"),
    ("cusparseCreateCooAoS", "hipsparseCreateCooAoS"),
    ("cusparseCooGet", "hipsparseCooGet"),
    ("cusparseCooAoSGet", "hipsparseCooAoSGet"),
    ("cusparseCsrGet", "hipsparseCsrGet"),
    ("cusparseSpMatGetFormat", "hipsparseSpMatGetFormat"),
    ("cusparseSpMatGetIndexBase", "hipsparseSpMatGetIndexBase"),
    ("cusparseSpMatGetValues", "hipsparseSpMatGetValues"),
    ("cusparseSpMatGetStridedBatch", "hipsparseSpMatGetStridedBatch"),
    ("cusparseSpMatSetStridedBatch", "hipsparseSpMatSetStridedBatch"),
    ("cusparseDnVecGet", "hipsparseDnVecGet"),
    ("cusparseDnVecGetValues", "hipsparseDnVecGetValues"),
    ("cusparseDnVecSetValues", "hipsparseDnVecSetValues"),
    ("cusparseDnMatGet", "hipsparseDnMatGet"),
    ("cusparseDnMatGetValues", "hipsparseDnMatGetValues"),
    ("cusparseDnMatSetValues", "hipsparseDnMatSetValues"),
    ("cusparseDnMatGetStridedBatch", "hipsparseDnMatGetStridedBatch"),
    ("cusparseSpVV_bufferSize", "hipsparseSpVV_bufferSize"),
    ("cusparseSpVV", "hipsparseSpVV"),
    ("cusparseCsr2cscEx2_bufferSize", "hipsparseCsr2cscEx2_bufferSize"),
    ("cusparseCsr2cscEx2", "hipsparseCsr2cscEx2"),
    ("cusparseCreateDnMat", "hipsparseCreateDnMat"),
    ("cusparseDnMatSetStridedBatch", "hipsparseDnMatSetStridedBatch"),
    ("cusparseCsrSetStridedBatch", "hipsparseCsrSetStridedBatch"),
    ("cusparseCreateDnVec", "hipsparseCreateDnVec"),
    ("cusparseCreateCsr", "hipsparseCreateCsr"),
    ("cusparseDestroyDnMat", "hipsparseDestroyDnMat"),
    ("cusparseDestroyDnVec", "hipsparseDestroyDnVec"),
    ("cusparseDestroySpMat", "hipsparseDestroySpMat"),
    ("cusparseSpGEMM_destroyDescr", "hipsparseSpGEMM_destroyDescr"),
    ("cusparseCreateCoo", "hipsparseCreateCoo"),
    ("cusparseSpGEMM_createDescr", "hipsparseSpGEMM_createDescr"),
    ("cusparseSpGEMM_copy", "hipsparseSpGEMM_copy"),
    ("cusparseSDDMM_bufferSize", "hipsparseSDDMM_bufferSize"),
    ("cusparseSDDMM_preprocess", "hipsparseSDDMM_preprocess"),
    ("cusparseSDDMM", "hipsparseSDDMM"),
    ("cusparseSpGEMM_compute", "hipsparseSpGEMM_compute"),
    ("cusparseSpGEMM_workEstimation", "hipsparseSpGEMM_workEstimation"),
    ("cusparseSpMatGetSize", "hipsparseSpMatGetSize"),
    ("cusparseCsrSetPointers", "hipsparseCsrSetPointers"),
    ("cusparseCreateCsc", "hipsparseCreateCsc"),
    ("cusparseSpMatSetValues", "hipsparseSpMatSetValues"),
    ("cusparseSpMatSetAttribute", "hipsparseSpMatSetAttribute"),
    ("cusparseSpSM_createDescr", "hipsparseSpSM_createDescr"),
    ("cusparseSpSM_destroyDescr", "hipsparseSpSM_destroyDescr"),
    ("cusparseSpSM_bufferSize", "hipsparseSpSM_bufferSize"),
    ("cusparseSpSM_analysis", "hipsparseSpSM_analysis"),
    ("cusparseSpSM_solve", "hipsparseSpSM_solve"),
    ("cusparseSparseToDense_bufferSize", "hipsparseSparseToDense_bufferSize"),
    ("cusparseSparseToDense", "hipsparseSparseToDense"),
    ("cusparseDenseToSparse_bufferSize", "hipsparseDenseToSparse_bufferSize"),
    ("cusparseDenseToSparse_analysis", "hipsparseDenseToSparse_analysis"),
    ("cusparseDenseToSparse_convert", "hipsparseDenseToSparse_convert"),
    ("cusparseSpMVAlg_t", "hipsparseSpMVAlg_t"),
    ("cusparseSpMMAlg_t", "hipsparseSpMMAlg_t"),
    ("cusparseIndexType_t", "hipsparseIndexType_t"),
    ("cusparseDnMatDescr_t", "hipsparseDnMatDescr_t"),
    ("cusparseDnVecDescr_t", "hipsparseDnVecDescr_t"),
    ("cusparseSpMatDescr_t", "hipsparseSpMatDescr_t"),
    ("cusparseSpGEMMDescr_t", "hipsparseSpGEMMDescr_t"),
    ("CUSPARSE_INDEX_32I", "HIPSPARSE_INDEX_32I"),
    ("CUSPARSE_INDEX_64I", "HIPSPARSE_INDEX_64I"),
    ("CUSPARSE_ORDER_COL", "HIPSPARSE_ORDER_COL"),
    ("CUSPARSE_MV_ALG_DEFAULT", "HIPSPARSE_MV_ALG_DEFAULT"),
    ("CUSPARSE_MM_ALG_DEFAULT", "HIPSPARSE_MM_ALG_DEFAULT"),
    ("CUSPARSE_SPMM_COO_ALG1", "HIPSPARSE_SPMM_COO_ALG1"),
    ("CUSPARSE_SPMM_COO_ALG2", "HIPSPARSE_SPMM_COO_ALG2"),
    ("CUSPARSE_COOMM_ALG1", "HIPSPARSE_COOMM_ALG1"),
    ("CUSPARSE_COOMM_ALG2", "HIPSPARSE_COOMM_ALG2"),
    ("CUSPARSE_COOMM_ALG3", "HIPSPARSE_COOMM_ALG3"),
    ("CUSPARSE_COOMV_ALG", "HIPSPARSE_COOMV_ALG"),
    ("CUSPARSE_CSRMM_ALG1", "HIPSPARSE_CSRMM_ALG1"),
    ("CUSPARSE_SPMM_CSR_ALG1", "HIPSPARSE_CSRMM_ALG1"),
    ("CUSPARSE_SPGEMM_DEFAULT", "HIPSPARSE_SPGEMM_DEFAULT"),
    ("CUSPARSE_SDDMM_ALG_DEFAULT", "HIPSPARSE_SDDMM_ALG_DEFAULT"),
    ("CUSPARSE_STATUS_SUCCESS", "HIPSPARSE_STATUS_SUCCESS"),
    ("CUSPARSE_STATUS_NOT_INITIALIZED", "HIPSPARSE_STATUS_NOT_INITIALIZED"),
    ("CUSPARSE_STATUS_ALLOC_FAILED", "HIPSPARSE_STATUS_ALLOC_FAILED"),
    ("CUSPARSE_STATUS_INVALID_VALUE", "HIPSPARSE_STATUS_INVALID_VALUE"),
    ("CUSPARSE_STATUS_MAPPING_ERROR", "HIPSPARSE_STATUS_MAPPING_ERROR"),
    ("CUSPARSE_STATUS_EXECUTION_FAILED", "HIPSPARSE_STATUS_EXECUTION_FAILED"),
    ("CUSPARSE_STATUS_INTERNAL_ERROR", "HIPSPARSE_STATUS_INTERNAL_ERROR"),
    ("CUSPARSE_STATUS_MATRIX_TYPE_NOT_SUPPORTED", "HIPSPARSE_STATUS_MATRIX_TYPE_NOT_SUPPORTED"),
    ("CUSPARSE_STATUS_ARCH_MISMATCH", "HIPSPARSE_STATUS_ARCH_MISMATCH"),
    ("CUSPARSE_STATUS_ZERO_PIVOT", "HIPSPARSE_STATUS_ZERO_PIVOT"),
    ("CUSPARSE_OPERATION_TRANSPOSE", "HIPSPARSE_OPERATION_TRANSPOSE"),
    ("CUSPARSE_OPERATION_NON_TRANSPOSE", "HIPSPARSE_OPERATION_NON_TRANSPOSE"),
    ("CUSPARSE_OPERATION_CONJUGATE_TRANSPOSE", "HIPSPARSE_OPERATION_CONJUGATE_TRANSPOSE"),
    ("CUSPARSE_INDEX_BASE_ZERO", "HIPSPARSE_INDEX_BASE_ZERO"),
    ("CUSPARSE_INDEX_BASE_ONE", "HIPSPARSE_INDEX_BASE_ONE"),
    ("CUSPARSE_MATRIX_TYPE_GENERAL", "HIPSPARSE_MATRIX_TYPE_GENERAL"),
    ("cusparseGetErrorName", "hipsparseGetErrorName"),
    ("cusparseOrder_t", "hipsparseOrder_t"),
    ("cusparseSpGEMMAlg_t", "hipsparseSpGEMMAlg_t"),
    ("cusparseCsr2CscAlg_t", "hipsparseCsr2CscAlg_t"),
    ("cusparseGetErrorString", "hipsparseGetErrorString"),
    ("cusparseGather", "hipsparseGather"),
    ("cusparseSparseToDenseAlg_t", "hipsparseSparseToDenseAlg_t"),
    ("cusparseDenseToSparseAlg_t", "hipsparseDenseToSparseAlg_t"),
    ("cusparseIndexBase_t", "hipsparseIndexBase_t"),
    ("cusparseMatrixType_t", "hipsparseMatrixType_t"),
    ("cusparsePointerMode_t", "hipsparsePointerMode_t"),
    ("cusparseAction_t", "hipsparseAction_t"),
    ("cusparseFormat_t", "hipsparseFormat_t"),
    ("cusparseSpSMAlg_t", "hipsparseSpSMAlg_t"),
    ("cusparseSpMatAttribute_t", "hipsparseSpMatAttribute_t"),
    ("cusparseSpVecDescr_t", "hipsparseSpVecDescr_t"),
    ("cusparseSpSMDescr_t", "hipsparseSpSMDescr_t"),
    ("CUSPARSE_POINTER_MODE_DEVICE", "HIPSPARSE_POINTER_MODE_DEVICE"),
    ("CUSPARSE_ACTION_SYMBOLIC", "HIPSPARSE_ACTION_SYMBOLIC"),
    ("CUSPARSE_ACTION_NUMERIC", "HIPSPARSE_ACTION_NUMERIC"),
    ("CUSPARSE_MATRIX_TYPE_SYMMETRIC", "HIPSPARSE_MATRIX_TYPE_SYMMETRIC"),
    ("CUSPARSE_MATRIX_TYPE_HERMITIAN", "HIPSPARSE_MATRIX_TYPE_HERMITIAN"),
    ("CUSPARSE_MATRIX_TYPE_TRIANGULAR", "HIPSPARSE_MATRIX_TYPE_TRIANGULAR"),
    ("CUSPARSE_FORMAT_CSR", "HIPSPARSE_FORMAT_CSR"),
    ("CUSPARSE_FORMAT_CSC", "HIPSPARSE_FORMAT_CSC"),
    ("CUSPARSE_FORMAT_COO", "HIPSPARSE_FORMAT_COO"),
    ("CUSPARSE_FORMAT_COO_AOS", "HIPSPARSE_FORMAT_COO_AOS"),
    ("CUSPARSE_ORDER_ROW", "HIPSPARSE_ORDER_ROW"),
    ("CUSPARSE_CSRMV_ALG1", "HIPSPARSE_CSRMV_ALG1"),
    ("CUSPARSE_CSRMV_ALG2", "HIPSPARSE_CSRMV_ALG2"),
    ("CUSPARSE_INDEX_16U", "HIPSPARSE_INDEX_16U"),
    ("CUSPARSE_SPMAT_FILL_MODE", "HIPSPARSE_SPMAT_FILL_MODE"),
    ("CUSPARSE_SPMAT_DIAG_TYPE", "HIPSPARSE_SPMAT_DIAG_TYPE"),
    ("CUSPARSE_CSR2CSC_ALG1", "HIPSPARSE_CSR2CSC_ALG1"),
    ("CUSPARSE_CSR2CSC_ALG2", "HIPSPARSE_CSR2CSC_ALG2"),
    ("CUSPARSE_SPSM_ALG_DEFAULT", "HIPSPARSE_SPSM_ALG_DEFAULT"),
    ("CUSPARSE_SPARSETODENSE_ALG_DEFAULT", "HIPSPARSE_SPARSETODENSE_ALG_DEFAULT"),
    ("CUSPARSE_DENSETOSPARSE_ALG_DEFAULT", "HIPSPARSE_DENSETOSPARSE_ALG_DEFAULT"),
    ("cublasOperation_t", "hipsolverOperation_t"),
    ("CUBLAS_OP_N", "HIPSOLVER_OP_N"),
    ("CUBLAS_OP_T", "HIPSOLVER_OP_T"),
    ("CUBLAS_OP_C", "HIPSOLVER_OP_C"),
    ("cublasFillMode_t", "hipsolverFillMode_t"),
    ("CUBLAS_FILL_MODE_LOWER", "HIPSOLVER_FILL_MODE_LOWER"),
    ("CUBLAS_FILL_MODE_UPPER", "HIPSOLVER_FILL_MODE_UPPER"),
    ("cublasSideMode_t", "hipsolverSideMode_t"),
    ("CUBLAS_SIDE_LEFT", "HIPSOLVER_SIDE_LEFT"),
    ("CUBLAS_SIDE_RIGHT", "HIPSOLVER_SIDE_RIGHT"),
    ("cusolverEigMode_t", "hipsolverEigMode_t"),
    ("cusolverEigType_t", "hipsolverEigType_t"),
    ("CUSOLVER_EIG_MODE_VECTOR", "HIPSOLVER_EIG_MODE_VECTOR"),
    ("CUSOLVER_EIG_MODE_NOVECTOR", "HIPSOLVER_EIG_MODE_NOVECTOR"),
    ("CUSOLVER_EIG_TYPE_1", "HIPSOLVER_EIG_TYPE_1"),
    ("CUSOLVER_EIG_TYPE_2", "HIPSOLVER_EIG_TYPE_2"),
    ("CUSOLVER_EIG_TYPE_3", "HIPSOLVER_EIG_TYPE_3"),
    ("syevjInfo_t", "hipsolverSyevjInfo_t"),
    ("cusolverDnCreateSyevjInfo", "hipsolverDnCreateSyevjInfo"),
    ("cusolverDnXsyevjSetSortEig", "hipsolverDnXsyevjSetSortEig"),
    ("cusolverDnDestroySyevjInfo", "hipsolverDnDestroySyevjInfo"),
    ("gesvdjInfo_t", "hipsolverGesvdjInfo_t"),
    ("cusolverDnCreateGesvdjInfo", "hipsolverDnCreateGesvdjInfo"),
    ("cusolverDnXgesvdjSetSortEig", "hipsolverDnXgesvdjSetSortEig"),
    ("cusolverDnDestroyGesvdjInfo", "hipsolverDnDestroyGesvdjInfo"),
    ("cusolverDnHandle_t", "hipsolverDnHandle_t"),
    ("cusolverDnCreate", "hipsolverDnCreate"),
    ("cusolverDnSetStream", "hipsolverDnSetStream"),
    ("cusolverDnGetStream", "hipsolverDnGetStream"),
    ("cusolverDnDestroy", "hipsolverDnDestroy"),
    ("cusolverDnParams_t", "hipsolverDnParams_t"),
    ("cusolverDnCgeqrf", "hipsolverDnCgeqrf"),
    ("cusolverDnCgeqrf_bufferSize", "hipsolverDnCgeqrf_bufferSize"),
    ("cusolverDnCgesvd", "hipsolverDnCgesvd"),
    ("cusolverDnCgesvd_bufferSize", "hipsolverDnCgesvd_bufferSize"),
    ("cusolverDnCgesvdj", "hipsolverDnCgesvdj"),
    ("cusolverDnCgesvdjBatched", "hipsolverDnCgesvdjBatched"),
    ("cusolverDnCgesvdjBatched_bufferSize", "hipsolverDnCgesvdjBatched_bufferSize"),
    ("cusolverDnCgesvdj_bufferSize", "hipsolverDnCgesvdj_bufferSize"),
    ("cusolverDnCgetrf", "hipsolverDnCgetrf"),
    ("cusolverDnCgetrf_bufferSize", "hipsolverDnCgetrf_bufferSize"),
    ("cusolverDnCgetrs", "hipsolverDnCgetrs"),
    ("cusolverDnCheevd", "hipsolverDnCheevd"),
    ("cusolverDnCheevd_bufferSize", "hipsolverDnCheevd_bufferSize"),
    ("cusolverDnCheevj", "hipsolverDnCheevj"),
    ("cusolverDnCheevjBatched", "hipsolverDnCheevjBatched"),
    ("cusolverDnCheevjBatched_bufferSize", "hipsolverDnCheevjBatched_bufferSize"),
    ("cusolverDnCheevj_bufferSize", "hipsolverDnCheevj_bufferSize"),
    ("cusolverDnCpotrf", "hipsolverDnCpotrf"),
    ("cusolverDnCpotrfBatched", "hipsolverDnCpotrfBatched"),
    ("cusolverDnCpotrf_bufferSize", "hipsolverDnCpotrf_bufferSize"),
    ("cusolverDnCpotrs", "hipsolverDnCpotrs"),
    ("cusolverDnCpotrsBatched", "hipsolverDnCpotrsBatched"),
    ("cusolverDnCungqr", "hipsolverDnCungqr"),
    ("cusolverDnCungqr_bufferSize", "hipsolverDnCungqr_bufferSize"),
    ("cusolverDnCunmqr", "hipsolverDnCunmqr"),
    ("cusolverDnCunmqr_bufferSize", "hipsolverDnCunmqr_bufferSize"),
    ("cusolverDnDgeqrf", "hipsolverDnDgeqrf"),
    ("cusolverDnDgeqrf_bufferSize", "hipsolverDnDgeqrf_bufferSize"),
    ("cusolverDnDgesvd", "hipsolverDnDgesvd"),
    ("cusolverDnDgesvd_bufferSize", "hipsolverDnDgesvd_bufferSize"),
    ("cusolverDnDgesvdj", "hipsolverDnDgesvdj"),
    ("cusolverDnDgesvdjBatched", "hipsolverDnDgesvdjBatched"),
    ("cusolverDnDgesvdjBatched_bufferSize", "hipsolverDnDgesvdjBatched_bufferSize"),
    ("cusolverDnDgesvdj_bufferSize", "hipsolverDnDgesvdj_bufferSize"),
    ("cusolverDnDgetrf", "hipsolverDnDgetrf"),
    ("cusolverDnDgetrf_bufferSize", "hipsolverDnDgetrf_bufferSize"),
    ("cusolverDnDgetrs", "hipsolverDnDgetrs"),
    ("cusolverDnDorgqr", "hipsolverDnDorgqr"),
    ("cusolverDnDorgqr_bufferSize", "hipsolverDnDorgqr_bufferSize"),
    ("cusolverDnDormqr", "hipsolverDnDormqr"),
    ("cusolverDnDormqr_bufferSize", "hipsolverDnDormqr_bufferSize"),
    ("cusolverDnDpotrf", "hipsolverDnDpotrf"),
    ("cusolverDnDpotrfBatched", "hipsolverDnDpotrfBatched"),
    ("cusolverDnDpotrf_bufferSize", "hipsolverDnDpotrf_bufferSize"),
    ("cusolverDnDpotrs", "hipsolverDnDpotrs"),
    ("cusolverDnDpotrsBatched", "hipsolverDnDpotrsBatched"),
    ("cusolverDnDsyevd", "hipsolverDnDsyevd"),
    ("cusolverDnDsyevd_bufferSize", "hipsolverDnDsyevd_bufferSize"),
    ("cusolverDnDsyevj", "hipsolverDnDsyevj"),
    ("cusolverDnDsyevjBatched", "hipsolverDnDsyevjBatched"),
    ("cusolverDnDsyevjBatched_bufferSize", "hipsolverDnDsyevjBatched_bufferSize"),
    ("cusolverDnDsyevj_bufferSize", "hipsolverDnDsyevj_bufferSize"),
    ("cusolverDnSgeqrf", "hipsolverDnSgeqrf"),
    ("cusolverDnSgeqrf_bufferSize", "hipsolverDnSgeqrf_bufferSize"),
    ("cusolverDnSgesvd", "hipsolverDnSgesvd"),
    ("cusolverDnSgesvd_bufferSize", "hipsolverDnSgesvd_bufferSize"),
    ("cusolverDnSgesvdj", "hipsolverDnSgesvdj"),
    ("cusolverDnSgesvdjBatched", "hipsolverDnSgesvdjBatched"),
    ("cusolverDnSgesvdjBatched_bufferSize", "hipsolverDnSgesvdjBatched_bufferSize"),
    ("cusolverDnSgesvdj_bufferSize", "hipsolverDnSgesvdj_bufferSize"),
    ("cusolverDnSgetrf", "hipsolverDnSgetrf"),
    ("cusolverDnSgetrf_bufferSize", "hipsolverDnSgetrf_bufferSize"),
    ("cusolverDnSgetrs", "hipsolverDnSgetrs"),
    ("cusolverDnSorgqr", "hipsolverDnSorgqr"),
    ("cusolverDnSorgqr_bufferSize", "hipsolverDnSorgqr_bufferSize"),
    ("cusolverDnSormqr", "hipsolverDnSormqr"),
    ("cusolverDnSormqr_bufferSize", "hipsolverDnSormqr_bufferSize"),
    ("cusolverDnSpotrf", "hipsolverDnSpotrf"),
    ("cusolverDnSpotrfBatched", "hipsolverDnSpotrfBatched"),
    ("cusolverDnSpotrf_bufferSize", "hipsolverDnSpotrf_bufferSize"),
    ("cusolverDnSpotrs", "hipsolverDnSpotrs"),
    ("cusolverDnSpotrsBatched", "hipsolverDnSpotrsBatched"),
    ("cusolverDnSsyevd", "hipsolverDnSsyevd"),
    ("cusolverDnSsyevd_bufferSize", "hipsolverDnSsyevd_bufferSize"),
    ("cusolverDnSsyevj", "hipsolverDnSsyevj"),
    ("cusolverDnSsyevjBatched", "hipsolverDnSsyevjBatched"),
    ("cusolverDnSsyevjBatched_bufferSize", "hipsolverDnSsyevjBatched_bufferSize"),
    ("cusolverDnSsyevj_bufferSize", "hipsolverDnSsyevj_bufferSize"),
    ("cusolverDnXgeqrf", "hipsolverDnXgeqrf"),
    ("cusolverDnXgeqrf_bufferSize", "hipsolverDnXgeqrf_bufferSize"),
    ("cusolverDnXpotrf", "hipsolverDnXpotrf"),
    ("cusolverDnXpotrf_bufferSize", "hipsolverDnXpotrf_bufferSize"),
    ("cusolverDnXpotrs", "hipsolverDnXpotrs"),
    ("cusolverDnXsyevd", "hipsolverDnXsyevd"),
    ("cusolverDnXsyevd_bufferSize", "hipsolverDnXsyevd_bufferSize"),
    ("cusolverDnZgeqrf", "hipsolverDnZgeqrf"),
    ("cusolverDnZgeqrf_bufferSize", "hipsolverDnZgeqrf_bufferSize"),
    ("cusolverDnZgesvd", "hipsolverDnZgesvd"),
    ("cusolverDnZgesvd_bufferSize", "hipsolverDnZgesvd_bufferSize"),
    ("cusolverDnZgesvdj", "hipsolverDnZgesvdj"),
    ("cusolverDnZgesvdjBatched", "hipsolverDnZgesvdjBatched"),
    ("cusolverDnZgesvdjBatched_bufferSize", "hipsolverDnZgesvdjBatched_bufferSize"),
    ("cusolverDnZgesvdj_bufferSize", "hipsolverDnZgesvdj_bufferSize"),
    ("cusolverDnZgetrf", "hipsolverDnZgetrf"),
    ("cusolverDnZgetrf_bufferSize", "hipsolverDnZgetrf_bufferSize"),
    ("cusolverDnZgetrs", "hipsolverDnZgetrs"),
    ("cusolverDnZheevd", "hipsolverDnZheevd"),
    ("cusolverDnZheevd_bufferSize", "hipsolverDnZheevd_bufferSize"),
    ("cusolverDnZheevj", "hipsolverDnZheevj"),
    ("cusolverDnZheevjBatched", "hipsolverDnZheevjBatched"),
    ("cusolverDnZheevjBatched_bufferSize", "hipsolverDnZheevjBatched_bufferSize"),
    ("cusolverDnZheevj_bufferSize", "hipsolverDnZheevj_bufferSize"),
    ("cusolverDnZpotrf", "hipsolverDnZpotrf"),
    ("cusolverDnZpotrfBatched", "hipsolverDnZpotrfBatched"),
    ("cusolverDnZpotrf_bufferSize", "hipsolverDnZpotrf_bufferSize"),
    ("cusolverDnZpotrs", "hipsolverDnZpotrs"),
    ("cusolverDnZpotrsBatched", "hipsolverDnZpotrsBatched"),
    ("cusolverDnZungqr", "hipsolverDnZungqr"),
    ("cusolverDnZungqr_bufferSize", "hipsolverDnZungqr_bufferSize"),
    ("cusolverDnZunmqr", "hipsolverDnZunmqr"),
    ("cusolverDnZunmqr_bufferSize", "hipsolverDnZunmqr_bufferSize"),
    ("cusolverDnDsytrf_bufferSize", "hipsolverDnDsytrf_bufferSize"),
    ("cusolverDnSsytrf_bufferSize", "hipsolverDnSsytrf_bufferSize"),
    ("cusolverDnZsytrf_bufferSize", "hipsolverDnZsytrf_bufferSize"),
    ("cusolverDnCsytrf_bufferSize", "hipsolverDnCsytrf_bufferSize"),
    ("cusolverDnDsytrf", "hipsolverDnDsytrf"),
    ("cusolverDnSsytrf", "hipsolverDnSsytrf"),
    ("cusolverDnZsytrf", "hipsolverDnZsytrf"),
    ("cusolverDnCsytrf", "hipsolverDnCsytrf"),
    ("cusolverDnSgesvdaStridedBatched_bufferSize", "hipsolverDnSgesvdaStridedBatched_bufferSize"),
    ("cusolverDnDgesvdaStridedBatched_bufferSize", "hipsolverDnDgesvdaStridedBatched_bufferSize"),
    ("cusolverDnCgesvdaStridedBatched_bufferSize", "hipsolverDnCgesvdaStridedBatched_bufferSize"),
    ("cusolverDnZgesvdaStridedBatched_bufferSize", "hipsolverDnZgesvdaStridedBatched_bufferSize"),
    ("cusolverDnSgesvdaStridedBatched", "hipsolverDnSgesvdaStridedBatched"),
    ("cusolverDnDgesvdaStridedBatched", "hipsolverDnDgesvdaStridedBatched"),
    ("cusolverDnCgesvdaStridedBatched", "hipsolverDnCgesvdaStridedBatched"),
    ("cusolverDnZgesvdaStridedBatched", "hipsolverDnZgesvdaStridedBatched"),
    ("cusolverDnXgesvdjSetTolerance", "hipsolverDnXgesvdjSetTolerance"),
    ("cusolverDnXgesvdjSetMaxSweeps", "hipsolverDnXgesvdjSetMaxSweeps"),
    ("cusolverDnSgebrd_bufferSize", "hipsolverDnSgebrd_bufferSize"),
    ("cusolverDnDgebrd_bufferSize", "hipsolverDnDgebrd_bufferSize"),
    ("cusolverDnCgebrd_bufferSize", "hipsolverDnCgebrd_bufferSize"),
    ("cusolverDnZgebrd_bufferSize", "hipsolverDnZgebrd_bufferSize"),
    ("cusolverDnSgebrd", "hipsolverDnSgebrd"),
    ("cusolverDnDgebrd", "hipsolverDnDgebrd"),
    ("cusolverDnCgebrd", "hipsolverDnCgebrd"),
    ("cusolverDnZgebrd", "hipsolverDnZgebrd"),
    ("cusolverDnXgesvdjGetSweeps", "hipsolverDnXgesvdjGetSweeps"),
    ("cusolverDnXsyevjSetTolerance", "hipsolverDnXsyevjSetTolerance"),
    ("cusolverDnXsyevjSetMaxSweeps", "hipsolverDnXsyevjSetMaxSweeps"),
    ("cusolverDnXsyevjGetResidual", "hipsolverDnXsyevjGetResidual"),
    ("cusolverDnXgesvdjGetResidual", "hipsolverDnXgesvdjGetResidual"),
    ("cusolverDnXsyevjGetSweeps", "hipsolverDnXsyevjGetSweeps"),
])

PYTORCH_SPECIFIC_MAPPINGS = collections.OrderedDict([
    ("USE_CUDA", "USE_ROCM"),
    ("CUDA_VERSION", "TORCH_HIP_VERSION"),
    ("gloo/cuda.h", "gloo/hip.h"),
    ("gloo/cuda_allreduce_halving_doubling.h", "gloo/hip_allreduce_halving_doubling.h"),
    ("gloo/cuda_allreduce_halving_doubling_pipelined.h", "gloo/hip_allreduce_halving_doubling_pipelined.h"),
    ("gloo/cuda_allreduce_ring.h", "gloo/hip_allreduce_ring.h"),
    ("gloo/cuda_broadcast_one_to_all.h", "gloo/hip_broadcast_one_to_all.h"),
    ("gloo::CudaAllreduceHalvingDoublingPipelined", "gloo::HipAllreduceHalvingDoublingPipelined"),
    ("gloo::CudaBroadcastOneToAll", "gloo::HipBroadcastOneToAll"),
    ("gloo::CudaHostWorkspace", "gloo::HipHostWorkspace"),
    ("gloo::CudaDeviceWorkspace", "gloo::HipDeviceWorkspace"),
    ("CUDNN_RNN_RELU", "miopenRNNRELU"),
    ("CUDNN_RNN_TANH", "miopenRNNTANH"),
    ("CUDNN_LSTM", "miopenLSTM"),
    ("CUDNN_GRU", "miopenGRU"),
    ("cudnnRNNMode_t", "miopenRNNMode_t"),
    ("magma_queue_create_from_cuda", "magma_queue_create_from_hip"),
])

C10_MAPPINGS = collections.OrderedDict([
    ("CUDA_VERSION", "TORCH_HIP_VERSION"),
    ("CUDA_LAUNCH_BLOCKING=1", "AMD_SERIALIZE_KERNEL=3"),
    ("CUDA_LAUNCH_BLOCKING", "AMD_SERIALIZE_KERNEL"),
    ("c10/cuda/CUDAAlgorithm.h", "c10/hip/HIPAlgorithm.h"),
    ("c10/cuda/CUDAAllocatorConfig.h", "c10/hip/HIPAllocatorConfig.h"),
    ("c10/cuda/CUDACachingAllocator.h", "c10/hip/HIPCachingAllocator.h"),
    ("c10/cuda/CUDADeviceAssertion.h", "c10/hip/HIPDeviceAssertion.h"),
    ("c10/cuda/CUDADeviceAssertionHost.h", "c10/hip/HIPDeviceAssertionHost.h"),
    ("c10/cuda/CUDAException.h", "c10/hip/HIPException.h"),
    ("c10/cuda/CUDAFunctions.h", "c10/hip/HIPFunctions.h"),
    ("c10/cuda/CUDAGraphsC10Utils.h", "c10/hip/HIPGraphsC10Utils.h"),
    ("c10/cuda/CUDAGuard.h", "c10/hip/HIPGuard.h"),
    ("c10/cuda/CUDAMacros.h", "c10/hip/HIPMacros.h"),
    ("c10/cuda/CUDAMathCompat.h", "c10/hip/HIPMathCompat.h"),
    ("c10/cuda/CUDAMiscFunctions.h", "c10/hip/HIPMiscFunctions.h"),
    ("c10/cuda/CUDAStream.h", "c10/hip/HIPStream.h"),
    ("c10/cuda/impl/CUDAGuardImpl.h", "c10/hip/impl/HIPGuardImpl.h"),
    ("c10/cuda/impl/CUDATest.h", "c10/hip/impl/HIPTest.h"),
    ("c10/cuda/impl/cuda_cmake_macros.h", "c10/hip/impl/hip_cmake_macros.h"),
])

CUDA_TO_HIP_MAPPINGS = [
    CUDA_IDENTIFIER_MAP,
    CUDA_TYPE_NAME_MAP,
    CUDA_INCLUDE_MAP,
    CUDA_SPECIAL_MAP,
    PYTORCH_SPECIFIC_MAPPINGS,
    C10_MAPPINGS,
]
